﻿using System.Collections.Generic;
using System.Threading.Tasks;

namespace ShopifySharp.Tests.Test_Data
{
    class ProductImageTests
    {
        public static ShopifyProductService ProductService = new ShopifyProductService(Utils.MyShopifyUrl, Utils.AccessToken);

        public static ShopifyProductImageService Service = new ShopifyProductImageService(Utils.MyShopifyUrl, Utils.AccessToken);

        public static Task<ShopifyProduct> CreateParentProduct()
        {
            return ProductService.CreateAsync(ProductCreation.CreateValidProducts());
        }

        public static async Task DeleteParentProduct(ShopifyProduct p)
        {
            await ProductService.DeleteAsync(p.Id.Value);
        }

        public static ShopifyProductImage CreateValidImage()
        {
            return new ShopifyProductImage()
            {
                Metafields = new List<ShopifyMetaField>()
                {
                    new ShopifyMetaField()
                    {
                        Key = "alt",
                        Value = "new alt tag content",
                        ValueType = "string",
                        Namespace = "tags"
                    }
                },
                Attachment= "iVBORw0KGgoAAAANSUhEUgAAAaAAAAD0CAYAAADdXvIMAACkC0lEQVR4XuzY/0+VdRyH4df7Oc85HDgkmEeDI8opD6CgIAQBIlna5kR/cDpzOkXUHKD2xVZrrbBZtKYubWtmqc0vqdNlbpab0CgVK9z8MihJoARkcAxhICaInPM8T+953ChbX+bqwI7va7v3+hPez+cBg+fwdIULgp8IIYQQCnzSuVT4mRBCCDlAk7ko+JkQQgg5QGmcAj8TQgghByiDUxGAhBCiKMUZtSTaPqcwNvLp/DERyRgchOfwdBNncCsRYIQQ4p2sWNc0W1DveMBI5DLNJmPmUFt1XrR9zdvpLhMGjFC4IfAZhQAjhBDnL7Vure++FWQzEUJUBR6vhobO7vizl9s3napx//z8WMcCDAihgMgGwwB0fQICiBBCrJ8cN7q57fpTNwA0aAZavTqCTIRws4JQVUFTV4/zeI17//zI8C/WpT3ihF8JBbpOMJtBFstkBBAhhLjs7lhepRnInzcXmzesR2bCWNR4Dei6AQIQZlZgUQiVV67N+vL7ptoCV8Sz8BtBnoNTR9IwezOpKnR3y3jzwpPVCABCCJEA1I1LeTTm03Nn4WNg1YwZ2FdSigQTQScCA3HdXh0ab9yIIaXJ0faCojP1jfhfCYXr4m5QWBhAtAADSIibW7Id3CRu1u39IDsE90CIF2OHx/YCMe8d2I/bDIOHsOVYCaaNi0OtZkCFj8HZVAU2E+Hi1evTS/k19MI4RyH+E2JdkpNeT3Km4C4E5jmS06BOnOjUqqvb1JwjIzBwhBygYJ4oLp5LBMHJO4zXzfs5dyK48FQv/oEQOQpejkjL2PDx6Qrcpt3iIcBkwbWWZmREjYIOYLhZgW7gjv7XkMEbw6+heMeDq4orGy/hHsmHgGPZDy0dRe7uXqcC1MxLcs5ZW9V4sf8A7X/iG3XqlCyjrQ3ec5WLLHnf7cMgIAS/gEbyzAZhMW86QGCnuaNEdIq3wlpw0oO7CJEClL22Y+e0OcvzAMPL6YDuARQLZ8bRLe9j4ernkKAQwP0eEeDVDFzXDQyzqJ5010MrN//YsgP/mtiY6go509S2p+5q19xOAMGc3W5HT3t7z7LMWMfqirouAuv7JHuXmpa8hKIc8B4r+8U8rywSg4wQN7dmxwD0KoClYEQEdgWEct4y7itr/skGOQDipdih1qqWns5Dza3WIeFhgO4FoPcfIfMDYFg8Ph4l1RcRpyrQ0O/u15CXN9ExdC8fotw15TUG/pYoTnk49Xit+7OW7lujwdq53du3YUZuLmZOSIbRWPdRSZ9WoMCnyrjaDtisMLmcEX27st7CICME/3r7KbiwfBmA0dxe+ERy87ntXH3vtilfc6s4F+5bwt3c+bgrPdN3fBgIf6T1gOGN3Xtg4b2p6WB/YnA2s4IQhXDe3bmopKrpQvFjrjj8JbE2MTrvcGXjmSt8fEIVQjOATW++gZxnVoAsQXjl3Q2o79OWbkwdYyUwz4EnJyE09Fs1YyJgMUGrqITe2jHBklt+AYOUEL0fTpnNsxOEcNwBQj9QKc9BEA5ZV5z4FfcN8Rt7ZwElx3Us7K/u7R5cFjLYkhESM8bsJA7DC7zEYWZmdJIXZn7hF47DjI4hTgwBM1uSxazFwe6+9c8/0+f02RmNd1fSSit5v3Pq9NDuxvamv626deueCx966nsufevL3vtuUAc40CRwVUh1AYa3P/pivvC7P3BskgW1IIAC/aGjL+WF5x45/6kfunntzxnFNC9cPPNj16/d/kaAbs9wY+h4/VP/i/f96MfEUCkWOL+vj1Nm+hcKQPDTh3sEYb89/qgOWTQbdu4kvP6OLVoJFqeecUWVaaYByl8/NwV0ALOAmSgd9Sv01EPJAp1Ath5KHghBc0CGhAClCFSAHcBOlO3A/cAWVf3/mU7/OCU0H+H3wHEtAhIDzqHVyg7x/cvwvW9kXnDFvzjomeZsuPqjl1/xsFPPPxc0ArRVQGLB72TzqpU87NDl5ICcZ3ggjMBg4PCAUw+Z89avrNryEabhQyce2nXD/Vt/dPeO4UemgU7fcGfgOG3pYn61eg3NPPmw5bi1K9/kAfhP/lNY/fbZV+qO/sfI7C7I+tjjls6J/rXyKuB0HhRMU/7GeZ3AYmA+cAjKPGARaC3oAxbGgsnRjLZ/vhufLZa+dPYq4C7QfwH/Bv5TK8HtpIlaA8LGWrntBJT/IBxHAoQhks9hF86fEW3Z+jLdsfNl5S8/7M+kvC9lnn/Fzzkomea1h3Tl7i/ak4497VTaIhaiKngRcw85lGf915P4+E9+xnFARHucNjavFgPHH1dt+fCLl8459Cv3b3kxD2Le99Blp//lrvWXbShUFnZaIWUMm4OITuArf/wTu8J0z0JYOUeIqf7wgpdINvVl78TDQRz4gm4bILxl/eWp/778Qpqo/uhCST3tL8oBxzSVb563GFgBHA4sA44CFqqyFOgaLQaduES0+Y3xfhZU29qpAFyH8mfgj9mX/+0mEqhJqBPYCHSMulsYwc6ZjVm6GC2PEN58F26ohOnM3IhvP1sT0bc4qJjmkhzndJ5y9pVfvOIqgF1nQBo1BGQ8SHWz8b57OXvFYXQAWc8wFiJQDh0bFb73rCN/cuF37nwKD0Jeceic19Uyn08WIqXLMwgQOeUOp/zw85/h0a94NbvimeecT+HqK/7PECMp76c6XEK37ICUhZEiMiOLd+ycC6o/uuja6vcv7CUBRJjaTFP51vm9lW+dd0YtXliLT9TE86darIlLXX8BvgC8EXiUKsftFfkwpnwmgJJAHuUC4MPAjbX27Ftqe4YurcXRAJkXXzWM8DgSwFoAwpWrCf5zC9LZQfoJp+OfvAQcx7vh6Jvlr5x3Ry3zu4SDhmk2FDl5yTHHkSDsEmMhKgOO+ctX8JRHPYL7AcPYqELWGqrAtXcP/pd+9by/Ff/n1DQPEj552mHznjK769d/Xbnlk9VI61khgAXucspzzz4zlk8bXEQIeUOM/6Q/bsezf4027IBSGSwwUkJ60njH9p0mWXt79XsXnUUMTnuZMkxT+fb5c2pxfuX/zn9dLb5Zi38C64G/A19FeT1wEbAYENowlnwms/SmqhP52mOBdwO31SR0eenzD3tG5kVXXYHwPWJQBRGkuxMtFKj85grCf9+Hd9wc0g+fi7/UA6dHatF8p/z18/5R/uZ5j+CAZ5oKnHzocccnpmgPCmhlCIAXfOCDdAGVyDEeHJAB+gtVqLizjMiNhQ+cOouDnLcdveg5v7jp/ntu2Tr0mI54wrgqGIFNoWOJET78s1884A2hOjxIAKFHApL2vqyDpfN1005kSQ9UIihUkZzgHZWfF91f+lv1+4+4NPWMP7439fS/7KyV7bzU0y8P2W2mGXrdidL1qX/rBGXTAxwDnAAcDxxXf66kdkciqmPUxiYsmInTKp8JyAjORzi/9LmHvYJIt2JFURFicIpk0rXwCe5Yj9u4g9QZObyTStj7y4Tr80T96dMR/lDLEC8D3px+3hVrOCCZJgWnHP7QhzAuxItbsrtZfvwJPOrkE/jZP//DCgGnjIkBKkEIziHCkcDNhfefekb+Xdffz0HGex6y7Mg7Nu74xK9vX3cxQJ9n0EQxhJGyFfjJ/36J/IyZoNo2+dy+rZ9Du2XAkADKTyTtDUfr+mG4CCkBjaBYBeOwhxjsIn1P8JNHXh/84OLDUDmqJiHLbjHN0GtPyIMeTnuofOf8VC1OqMXza+L5Qi2ujTObvwGfAZ4bi6i9fFSBJMDFoWjy3l4rve351+ruiU30DI3cE1Bk15YVTE8GV4yoXDmMuzdAeqv4y0bwFxeQdIRWzVNRubsmotdxwDHNKxZ3zO+c07d0xTFHMx5EAHVQHQDgWW9/Jw5wkTIeQiDte2AMqgDMiyV0DAcJnzjtsN7nLJzx4T/evvaOW7cNXZwxQqdvUBI84B6nPOOUk3j4C18CaBxCMxpFlIaHweh2A0BShlN8+yWtRESr+0FcI3BQCaASYeZEeIeUT5He4G4k9UOi1OerP7wwwwSZls/xngbOdzuDCglUvnvB4bV4SuW753+kJp0/A2vjLrCvAy8HTgPyLbVtpUkwcagDtCn2cemNvVl6m5DoEiR5X/IGxVK9o4toTSeEBtMR4C8oYHoqqJImkk/WJHRFLVZwwDDN+o0jx89etoJMNsO4MR4ajABw5hOeyJlzZ7PJKSJjy6sA5NIe+CbJ4qEL+HdNQqdwAPP3Zx7X9YwFs9/521vWrLxh/Y63lMOIXs/gGUF1tMR3ho6ZwPsv+yGxZWjHjs0bMeUhtpTTRUMzqh+XjIfbVkY3DkNGSeYoRVB04IfYBUN4i/qPNN3VF0D6pur3L35F9QcXdvGATDP0yofK0CuPzwPLTKe32FuYOb78tXO/X/3pw39Y+c4FceuxXobyZuBCYA4AKq0GIJaLOpLHY0gmkVCS/SSxh6U3YZfopJXeWt8XxkZB0gpWCe7PE23LQGAgEmx3FdtbAs9BJOcCd9YaOJ7HAcE0hZCj5xy6golhQEMIBgF4yqtexU5AnNIOBSIHp3TkOWJ+HkRp+ngKuK7wvlMfBlC7pjhAGHnLSTP0Iw9713/u71/5qw1b37+1WOnt8QzZWiitiFPuB97x9jcza8mhoAEgtGPbho2sCyIuPLp3tQfQlAVtC37+iO+Jb54ZrinjZx10KBQiIAKNoOoARTqGsdkR34x0rHBD+Q9qMfX86vcf9RckvAxxt9XWiSo8yBl+4wkinvHxzVLJmLNMxj5csvZcMnamZIyVjIVQ0WIEQgutJTSSa9ssKL4qbRBU22U/sptZiAC6f0pvAKoT+xoF8RVCR7ApizdbMNkArVjEc5juCq7gQ8WzGP1GTUJnpJ97xYuY0kwzCMcsWrFi4mm5eGhlAPG7ecLLXsGn3/EuBiOl05M2TVxKWZWXnjmP84+bAYVqu7z76pp8lgM7C5ee+i9JswHHZldhA8o24D5V3SLCfR3vu2GA/cjwW05aALxCaoFK17r+MhlolNu0/cbcNaFy1uxZPPd/PgI4UAWhLRtWr8YBO8t2nddG7+/Al2dSUcKVAd5hQDqEUiwgItAQKhaAmoiMyRa6tJh9qBbzR2gp9Uyt+uuqP3zkB5DoJpxuR6mmnnFg7xuqdbgYnFoUo6GKOhTneoA+RCpiTb/4iNTkjW9WSNrUpGMXSMrMkZTNiy95jHiAaKTUo+zAKQiANolDx64rqU7w5i3AuOUzTqHI/iq9tcrHSOLCsVDAUyRSwu1pvF6N14EsoA0hiaJVH5RaK/v5xyE8Ov2cv25nSjKNg6OXHnYYE0YshGWIinT09vLIJz6WL/z81xwFOFopq5Ixloc/tId581KUB6u0h5uBx+HoxudYb65Dq+BGpH7VAAgpjFx6yhqUlSi3oNyIcEvHpTfcyyQz8taTlqnyKuAVQMop4MHC3jQhINpe5WGkFIFLa40HdcISmNQDVkRW3X0vBnjooq5VQhuCnz3iMoSnaBkko3jLqyABlCOgCi4CBFUBBVTBOlADVR8tZ5xW0lWtevdoaP6B6rfBrUddBdUQJaCOlkEsgmo5UhyKQVQBVQOAIigRioAqShygOIgvgAKCKvFz4uej79PxV6KQRPwgAhScIkROUASwKILgiQGsASuINSq+vBzf/I+kjKuHLyqeEaxYEIuqEMsGpwAomuQMzQJRRqMuEQPaIp+JZynsvexHdz/7UdXxZz86zuwnUiRtwQqojPattHcwIoBFK1VsdwXJWAgk+W8VGjRIgQoY3YxwUfrZf72NKcU0rzpsjv3PPVu2f+Vf/+456sQTAE0ye1zrRtQ41EWgIURlwCCdS7npqqt49LnnslDAWEMz60PH2Qs6+d6bHopWQ4oVh8gD/u72A3cQciZZsDMVSTm0JGgotQACGlcHGjRChLsQrkO4Ari8liVt2IviWQG8FniZKkIMCh1dPnevKfCYr97SyIJay29Y4M7Q8V8nn8z/3nADuDKogthGYECEZt7wtGfw18t+sPYmWOLRnteiPEWygo4o4b0Gb7mDVBnKNFABJQ4DgaWOCTEdw0azxQxV/zgNUrXwX6qh9wsi/684ClpDcFlVvR80hdAjvqlqNRpECXH049QpCAqiEAuIWD4SPxQRIAKgAgypai8O4s8LWqMhsLBJQAYQUckjGmBkEZY0RsRYEYw1Ys1CsZLDGiOevApPOsWLBWQFjBDLwKKKRg4NHQRtSmGqCDCu8QCqbW7usrvZwsFbenPJHwYo7ZF2y2IBdsE8tH8nVGuRz0AAIGAc4gVo6AMyF/TG/7/nqiahvzFlmGbVusFl3TM6exavWM5uYTyojgARDz3nHE5bOI9/rd/EgqTQgBfPgtsMnHVEH3T5lDZUEWEseoEzsSBWcUMgnkAtNAAiQBWMgComA9IDWpIjXIkjUJ6LEo68+5TfAT8BfliTUbBb4nnbyUeBvhHlebShNBRw+OE9vPD4ebz7xk2coIqIoCQUI0cOeNtX4+wnLINNj1kRue/2O1gAd94EeCQ0rwVtrGVBX6FSfbFdOAc3EBDethnvCAOpClRSTVmFJDcCDBoZUAUvwPgB6gwa2idoLYg8pBbqLOLMCCplVCCDknagWhCnA2gNp0lGo6qolnEUUK2gdBgVDyWDh9JgUNAsDhBAJAfaRR0BERDiYBZGOsUIxCFWQIifg4g03agUddq4hg6U1gV8tE2WQMtnBABtsx9HaEWaTbKHpTcmWHqTpudTpPQGiG8YhTA+BLQaggvwjz+a4F83w/AwkktDJOAEJJZQ5APigV5dk9CFNQldzqQwTfGTZ6aANGge8AEHCIoCI0CQe8M/isTM7yybTfnFdHR1sduIgcIGyC/m0S94EX++9H0sdIqKYAW2Bo45GY8LsilOWdEFpQCE8aOgTjAdDi0IVJO7sDpJkrUIJAWmT5ESuCIQiKcRj4N6fGjkXad8FfhSx/tv2Dou8by9Jh5IxNMeAgV/uMJbnngI60Yq/PTenSwy4BlBAQusVHjV4x7H0oecCMEgiDemfMJKhc333snyTrmDYcXjgXk98AItl6330MMJb6gQ3inYFYL4Faj6sRhMU7ppEimpQVVQBFwZkRAyGVCLOAFnOzQyHTjTeE0NqJkFAli0uQNMmx44mtAkaZAkWxBiBLR9xoGqgnPUiRSNpYOMuU+mfTOACBDLSpL+eLRdWUxAJyiG9iSam7A0hLHR/dX1Nlo+KQsGUCaIACCZNNHqdZhZOVLnHUf1LzehhQqS9YgzfFQBE4J6xPylJqEzaxL6B7vFNMVPn7kAOCqORcAKlD5gFkovkAUygAe4JOWmBFSLHz+jX51uzy3r3PDtz/3Z+11lIXuESTWaEfKLecwLXsonawIqRErOE0qR0g+858KlLJmV46hFWaoDVSaEbchEKwaTU7QMiEBKQQBHfIVopyAlMHlFfFAUEdBIQFkAvBd4U01EH0P1Qx0f+Gf1AcVDq3gQASPgtBExRqBYCOma5fGWRyxj/UDImv4RRBUjwlDomAO8+jOfpE4YgGcZi1W338qWapWzFs36N8PbYgG1z4IKwc8f+Wa3efMnzML5eA8/meDXfyO828cuKyDZAlT8pu5eaXosqAIuQroXgyq69X7QCKwPno94Bkg+j2tICxUklhiavA9xKKiNbaNCIpLmhENRYhygTXJQRwNFWpZkFIRYqm1kB0iyKAVC0hIt0npDR0CFtuiY2c/eWPdpokXyYz3fv6U350DirNUzxKULdhtVpCtP+O9V2D5H6tx5VK/ahJYiSAs4ISECPEAB/n8mdFRNQvfwgExT+uxZHspJoBcCj1DlLCaGJaEDAKVPRA6l6tiyXZl37FL2CAFcANVtzF40j3POOJWf/uN6linc4ZT3njKfZzx6EYUdZbIpKJQjRIRxo2A80Ai0LCAar/0I4muiV0BSoGVwKuApRKAqNJGPRfT8kXee/MaahH5MIp5DgLcBLyQGAaxBLOBAyyFaisAzkPZAAaegYAxUhwP8tOGIGVmu2jbEIqDHE+4H3vKMZzB76aFQ2QI2zXi49d83UgXOOqzvn59cmQioLf4T//DJ4LILXhvdfucib/5M/IefQHjFzUT3GewCH+keBqcQ2EQQo2QR4yIoDyOLHoIsPhbdshIGtqGVMlSq4GL1i0HRpJtJhDoGoElygCgJOvpnapIFtYoJYrE5UGLa3MQEBJf8QJoFkPz1AqZJIkILKu3WZSah9DbWZ4VxMhVKb0kWaUy8BpdIeY8xBnyf6g0bSZ0t+MeXqd6SgopBUgoqiSmJAAuoBf5R2zS8NP2sv47wQEzjAX2xwW8F7gG6gFnAcUAv7dA2+71MchNYswXmPHIRu406MGnIzoOBVTB7FhfXsqBv1wR0Y+R4/uEzecvzD4egStpXRsqKEWG3MHGZzQhqAKdoNVm7B0E1/lwAwphb9RYDl4284+SvAF8Ano/yakAA8AxSCyKHFkOiYrUhHlFMygIeaAhGQASsgGepRMrcPp93PHEx5g/wrRu3cG/omA+86IMfAEDDCPEZn4D++S86YMuTfn/3PQAe48H3n6nDhavdLXdizjgW74xDCf9zP+E6gy2nMH2D4IUQpEBpLS0pYNPo8A70zquQ5Scjh50C1QKycyNaHIZKASoVtFpBwhANHUQKTkCJQ0bLRElAW+TTOrU7yVASmTgABE0clGRISQHPKZhdiw4EdYAmkmi/XjKGfLS9fPa89CbE7GHjASBKC+0b9va89CY0EEGMAXmg5TJhdxAUsgY3YohuG8IeViB1SI5gdR4NDOIrIBDFB5r5mViIzAD+CpxCW6bJvvqaMvC7eowuwz0ReN8oASngkiICphEi2lIM8KzAUJU1/XDq8mXsPgJhCelcBjYFOB72pCfR94Ln8ahD83zxNcdCGDLSH2CsYIQ9Q2i9VUSACiJNPT6RoPHjMXhxPWJJS8oDFFcIcCNVdLgWhQCtRGANJu/hyhFaLUOkxPJBPIGURfIphiKlq8Pywf9ewWOPnc07f3IrD3vkk5i9ZBmU14HJMF5uufY6DhWuvVchEdDYWdDfgp88/JfRxm2PN/etgYU9eCfMI7pnC9E6g5ZmYmYOI+kSRAbUa92U7xTSeYgC9Pa/wcb7kCVHQk8fkstAkIeoioQlCMqNCKv10KgCkYPIoC4pyeHiaJaCkpS/FBBFFFSbZaWtfQNOQJqE55rWZRRoK4528olRdoc9KL3J2ON2BEBAWsuXaNNjUUDbi0pbZYRr/r7jkI+MIZbkc3sXBZNSom0dmE5Fsg5vfplwQwYNBTERkstBFOH6B5DObhAF5eRaFvTpWhb0Wh6QaUqfOXM5cIEqjwTOBvogWXwHQawiPvUrVmPxgCYDQMAJAL4VioMV1jqYu3gRu40IaBUtbULyCwHo7uniNx97Ecfu+HVDPgNVjBX2KqpNTaVNuy9InqOMDysQOKItIxDLpy4dI4g1SCa+RxfD2FQkRg8UVdCoimgRmZVjeH4ebyTg9MOzXP6s+fCeDwGKBgH4WcZDZWSE22+7leesyP7jD/eWYgGNF2uehTIQrd5sbJcHotjDupHcANGaiHBDH7avgMkXwIZoaGMhNGUGxoN8Dzq4A73tWmTOQmTmTPCAagnCMrgqSAQ2AlNFtDr6l06F5DmICqqGpItEETWjRREJ6K7ax0e/rrrrUh+0l4+6Nus3Ey296aR3vSVlTTGAJrJ2CpGiScdh8l7LrFIFaS8QEWIEhFak+SpMKsLE8BUtW6LtGezMMlhXu1YIa88JQUsl/BNPItqwgeju+5DebmJeU5vh9+v0Ja2dcUyv/ywFHo3jyaDnEUMEhIAAKcXkFJMCPJI/HONtOkStZSgByhWHNQEfveQwjjh8GXuEyUB1ADIzwWYAOPbZzyX46E+pDgWxfJR9QOtatjAxjOB2lnDbS5iOFElDTSI7PNO2SiM+DRkNVDCzMkRpn8E776br3P9G5h2GFlaDzTBebqplP0WUY5cvuJp770sENM4saLjWlv0S11/8qlm/HVnYAcUyZq5FslWi9RHRzk60mMF0jUC6DNZB6EHLOg2QyUMUohvWojt2IDN7ka4UeA6qIYRVcBVwAThBVUBb5UMsn1gCo7vyHPFVWlvGVRtXxpCPktBOPokkJrDWMlmlN2ktvRkBAzjAuXjRM4Aolo7T+L022ZG0voAZZwYiTB7C5KAgvsMN+0g2AAHxFNtVxQ2lcSMjRGvuxzv/NAgrRKu3IF1ZEAB+Vhsm25e+5PKIaelkgP9CeSrKYxHAKBpGaAXEWKRDMR0Ok1UkpWCAEFwgaFXQCHAPnPG6CCgWecRjjoG4BLdnWVAE5R2QX4AW+yF3GOGiU2H7/ZDtZp8yqorE+IkUrME/YibRxuFajCAAvpmgP7WRLXkGQoeow53+31iAoAypdHshC6O4+q9X0wEDS2d3/hOYgICSrrivBZdd9IJo3dBpXpdAKoKRKpIJ8ZZVcNsKuO2dRNv6kHwRkyuCX42lYUZnHigYC9kcWg1w6zdjchmkM43kI/CqEFRAHWCS7KdFBAZ17fYjAdrsheasrP1stNbspzVzbrMYMfbXjiknmUDprdUUCuAZQKHq0FrUr2FDOjgdnYmIIZbVGJMD2k8YAMb93sTFIuxTjKKhQQs+ko9nxPkOyQUY00l07xrM3A68C4+C31cb/wfv9ADpAr4MvIgHKaXPnfVQHM/FydMQnYsFoipaKIPxkO483iLBdBcxuQgRgwaClgxaBg0lKYU3h7ZKSBA0rFCpdJDeKzXYNFT66wKiWsKt/Td2+RlE625GYgFNeQQIHYpgF3UhaY9w7SBScZAav4Q0UkzOR1Ie0ca1eEefhz3qYVBaC14GkDjG5g+/+wsL4Krj/u9GR4xhovjmyRoo0ZoR0ABMCOUIggAzYwBv8WbMzH6opol2zkCHuyD0wEQgrjUbcoDnIekUWg5xm4tE6wN0BxAIWEWNAxKxoMTRJB/X+peCOmnNvlxTpjNmCU1Ax8hadPzToHWCmy3H3hhqRovBNwBoIcTtqBLVQvsDtBBB6JIacRxJx+F+yDxEmKqIUVzZQ6sGdaBVi3gR4oeQyxPevgY2rcE7qQM710dLxPDCyvcuOOpBKJ6nlD79sD8RyY0YXiNpN1cyjX9fZk4v/unHkX7MiWQefQj+WTnsoVUkFyZlOFUwgFEQkOb1R5dUMHAyeg04qiL5mewVxEJUgvJ2pGc+bttqotv/hOlbCCgHDAJEipZDzOw8/opeMKCVCGT8CSF5H0RgZCf+mU8HBC3saNLHGCJyIe85rsQnHz/vtyTgMUH8J9YnJLzC9bsvmE0RMieAahD3zntovV7ej+ZKuOE8WsjjSllMptxoUrABIOBsa6LgWbBxCr7Ta4gpE0I6RHwHxpGs5xhwiRhwraKgVT6NYPRVaVN6G/e6j7bfR6OTV3prtob4Bo0UNxDgiiEEDogl4yWC2jdlLuGAxyiEhnhCNjgBLGJDJJvCDYO7dyNmRYi3vIgGedxIBslEAF8HTucgp3YceidOX6ChvJSQwyWtSNqBdUDjZieZLszs2UhnFh0pEK7fCSNFtJqFSNHIQCDxiDaSaodrvIZq+8qEAtagGiH5GXuxButBaStkZmKPuZjqHz+K9CxM0rADCQUtBUh3Bq8moeCefrQcIf4Yo6tCB1kP6fBxg/2YRcfgnXgRBFvBy0yoq8qtvZ2zZw3Dknl/4JebdlNASSnui8GPH/G0cLM928tESLYEJYciEBjUGfADbN9AQ0TFLFrKodUM+BVMqgw2QFHANO0fAixgFNSghRTUQr0ISUXgRUgtMKAoOCBK7Iu2W/dJImEvrPsw1rrPZJXeJJFjPJfOjYREg40WS6lLx0zMD8JByh6WBEMLRpOz/ZCGhFJpom0BpmcArOIvUqqr/Lhl251Wy4JOSz/z8usOUvHMxvEKrcjLscxsrOM48BQUNAACQayHBmWi2+9AixUUC2IBv3EVgJa9e6MzIHkgTxjcYLyXsGsWewUxYHwobYOuMmbOCuzh5xHe9VdM7xJAORDRUoh0pvEO6cFtHkED17hXpG2rRwQ0dNgZWSTj49atw3/82xpZ5pZrINXb3HEE0r7+Ht17AyPrN97R/eOV60jAY3fxeDxVtkUbsp63eAisg6qfCCC0qBrwQmzXMC5bQitptJzFlTrBROBVEC/ueFPbJBEBUbAKCoQGrXogDrUKnkNqgYlAtBFOQBL5tIigSTbqaCOPsdZu2pXeJrjuo+OQDw8sH/EFdUq4rYobDkFAfANykHhlfydgEv+FHiUTN8AADpMJcEUf15/FdAYAeLMKBJs6QQVEP3WwZUGlLz5stoa8Vovm1aQ0b2dEmI4Q8RUicBWTdKtZwDlEHZpNIelaKKCKRskmmKSZCGQ8p5BoIyTr4bYXCdf1481KY/ZWBuSCuAsuDQN3w4yH4NWyoPDmX4MAygGLlkOkM4XNdEPVEa4bQgsBkm5SgWvIXbrTaDWAbBf+CRcAEXTMR2w3WB8NhkGUsQjv/Rf4md9AiQQwJEy0FDcgKfcUrfhEm/tiCbimSQhA6KG1AEUyJUxXP6ZjEPECiFJopQNqgfNAFHC7FoIBvAiMQmSg5OOGU7XIoIU0Wk6hgW3IB8A4EAWY2LqPjrnuM87S216aKKA00SSfUAk3V3BDAeJJPQ5mRGTflgkFcAKRIVl/AJwFcSDa+B0MLa7gI6kQ21VuSAtOq37vgmM5CCh96WG9xc+c+UE3pKtE5W3ewmo+fXgZf1EF0+EgAg0FRMFTJOUQT8ECJGU1IkHDuHweP8eBKhPDM+CUcPVOcA6yOcjtxRJcdRC6V0DnEgDsMY/E9C2GaokDnsCBEaTDx1ve25BMOQQloRrVRUVHCrd9I95hp2GXn44OrkVLKcjMQKuDScr6QFSLRKtvRDpn/IomPIA9kNAvautBX3fD+Rfgh5jewYYc1IzaGKqO+DUa2BDJBIg2zlnRyIPIR5wXq7cebeQgYDQxhQoaeBAIoGh8UxCrsYDiEAUlyZDY7f0+Yzfm62SV3prkEyjh5jIaKpIy+yDlEfYpwtTBGRDX1HhiEC/ClS2m6AEOLVtMtoor+3UJidE3As85cMVztk8Qvtn1l95gspleb0UKb/4Qkq9AyUdHDOoUrIA6xAnqBA0EQsFFBiJi4bSZIzjRjS4KkrZEa/px/UXswk7ETyP5PvYK4kMwBP23Q+/RAPXvbQ85heDm32FmLOaAxylajiBl6xKK1gzhthaRtAUjAEhPmjrlAumLX0503z+p/vbzpC/5CFLeDFEZvE7GIlr1T9yWVdu7v7Pj73tPQMl60AuDnz7ybO3vWqE2wnQU0IoFJ6BC62nSAk6Sx7aK1AJnwNl6JLJyTdmGIX48KlEZNebAGVDQUECIhab1EBpX1ZYhcg+cCaGojndgpwCTUXpL1rnEEzRSwi1lNHQN+eh+EIvQgggHHwKq0hBOcxnWAKHUxUNKITRgHDZTJSxkAX16LQt6SeqZl5cPvHLbWc/X4cJ7JZdflDp1Cd6hWchvhx0DuM0GShaNBK0C1YZ0NDBotOtNowjNnT8TLGklpTcdLBOuG4xHzURIJl+LLvYOCjYHlZ0wtAq6DgHArjib4F8/BRFQ5YBHgGoEnsEu7Ya0RTeONO4p3el6UA0x81ZQ/cePCf/1G7Kv/zFmxlx0+431LAjRMfdhVK/6C27zzj/SCh57A9GLsO5+t7MHEUXSZTRKNc1wi6+ueeOnSYREBNYlm0kxjJ7pE8VPBUgiQWPZSPJ1CqhJRJDIBxVFACR5EwGaMh3VpkGUsQwnUnobWz7jnFYtgBGirRW04pC0AT1gMo8pm6WJjCVYAZfM4yK5ogquYrAeaCQQWTARxgshtCmMPgn4/gGU9VyIi96H43R76CF4h83FdEREm7fg1g6j5QxiHWJjGYcCDtQpoIgIagEHohrLW2ldixWYoH+wAkC4ph9CB55FCCHdieR62Xso+HkYWQepbsjMwB5+LtI1D4IKeCkOApL9Qk4x8zvQtMWtHMB0p8GzECp4KSq/+CrZl30C78gzYeu1jTUyFABBeCCCa69Ect2/hoG9LKAkC1pTK8U9FWcui2oSsn394Ado1Uvk0259RYUE05qmK+AERQBFUMA1byptf4aOAOiu94k6ixKTiCl5HNuodcimAXSMlmttHQTanrFLbzHiGdxQgBsJkZRhXzGNJLZpynhF4nbtMBkHhYDYCI0swFNiAU118SwEPgw8UyPBpD1Mqkp0551U1w2iDkwP2B6DpBUiIJCkWBElMxhxNEjaqlu2QqhO3AmS8YjWD+J2lJCcj5YDVB2S66pLaPdQ0DjUkQQgBnbeBnPPQPoWYxceS7T6eqR7LgcNAjiFUoj0ZLBHzAAjUIlABLd9Lf5JZ5F+3Osh2A6Z2aPvTUIrOMAQrboTt/6W0D9+7m8mR0CJhH5ck9CniMzrov5eTE8/2BACD2iaueZAm19T2XUZTBPPgkkyX1UQN3psrgKYREZC8lilSQ0ySk6tB5ba+D3bJAdplsYYPtE2Rz206aZrO3MeMIIGSjQQIFYmvsFTprvh9lxCbpelWI0a5SckKTMjDowDZy6qleFStTJclSlK+cvnvFlVL9VIMiiYvGD8MtGanUAKu9Bi+yIkDVQjXNmAKuIDRtFAIBKIDBoK1EIdTdLZAxRIW7RQxdVLbzaZjq4hkukEkd3f82MFoghE6tFAwcvF8ykrYHzsURcQ3vGnWEAHIdUIMh7xbEgwgg7vIPXCz5MMa7bgqoDXfv8pBoDw739BK0O/7/j4UGHySnCJhF5fk9CJOHO2G+jBdA02JBR6o0tvxGjrtUWnbcfdGFCTSEAdiCbP0aZzibTJ2pq814K02QAk7Vui0URs0CoSdYBMvPSmhhjEE6Kd1aT09mBG2E9Iq9lE4wzAxCVkASXpDMXkEc4ErpiC4nkYwqc04kQNBJNVTGcIQRGtCmZeH3ZegOkagkCg6KNGwCpUpS4irQhajcUTkQhHWvdLo7vvfrGGcP1go5U4n0Kd0iCEdNceDKJ2iM1Cug/EAxcQ2xPEgEkRA2EA1k/qrwcjoUuqzgNb8A4/Hf/kx4MGUNgEGoGfB4S2iAEg+M9fsXPTP4AKkyughAsxbg3OzHND3ZiOoYaEnG3a+Cnj2gCqCEDrZ9FRnwObOAdFVCFZ22kqiWlTViStgQNlnCjabBBtb5VWybaxkLZuDHPDIaiioWvymCCxC7HC/uPAngPXHh1zyJ06QUSajiJxIArwsKkkoPL/npMCPobyai0bsIo3K8RkCmglRGYtxCxbgPRUobgOtglu2ENHPFzB4EqmLh1CUAciJGuoBlD2Fsmen51F3NZCfJSAkpjJTXBGmzZCHYKiUQUNRsD44HcgqR6a0eGtBH/9PMENl2FmLAGEgx5VGN5B5lXfpIGFrmVoeQeEJTAAtP0rw23dSHTvtaE9fOEvYeUkCyjJgoLw1xefDXqLViXrhruQ3AjGq6JRazkOHqD01iwkZIzjoSXJspBk6nXbsbLalO0kmwxBxpH9MM4sRsf3WW21TgxiDW4kgEgxXT54BjGACKiiTiGMxRQpCogVEKYuIgdB6iWjmxQM4HaVSZuTp5B8Ho3weQJZqhWD6Q2ws+Nu1KgDWXgIZnYPlHfibt+E21HBFTpwxVg6AAbEAFYRo9RRJgcr4BS3YQgUsCYRkDRCUvlxSSfJaiz4WcAiGHZJUCK6/19Ed15OeOtvcVtXYmYsAy8FLuLBgHTPJLjhD9jlD4O0D5UNSLmApjrHPMwr+NvvcYNbftPzyS1FYJ9lQHiP/f19lf87/zW2N/UV1KLFTlymAF4l2SM0/tJbe7TdU21dcyH5uaBJjLZAknqrkKQUMtbPbY/qBLretP3PiBR8gzcvW7+qgCTCRnHgaMgniNCyQ0sRdazs2ywG4SCl/UjmxDW77syEY9jPlL9yjo/wORwv0REFX/EOLePNDNBQcEXB5DyobCe87k6i/ipaTQEpVAwCiK8g7DsUJO3hNg/j+ktI1gclQeLIdI6xHuolFREXJWdfO4dWh6HQjxYH0P51uO2rcFvvxW26E7flHrQ8gnTPw8w/CiL3oJEPIkjPXKpXfo3gn1eRuvC5pB/zaMgvR6qb0Ep11zcH4wEQ3PBHTF/u/6DIPhUQQLSz8lsthdh5OSRr0WIeSVvwSiAOnJlw6a31c20ObmubKSkJps2Bby55rEoio11PP1AYv43ai2zsPUWqiGdAYhkB2jpJFawgvo/ktCGhwQACB57QzHSDwl78IgEVQdS0G8+0tPr9C+aknnH5lv0kn/MQ+ZpWw0MoRNh5ObwjikjXEIz4tUiBSxFuKuCGh9EojboMYgCjiCgo+x4raBg1TvY0AgKJ1xWMgFFIddFKcwesARF0aAPB9d8luvsKcGHjtcJOtDgM1QIaBWBTSLYL6ZqD9CxIxPVgI3KYxUeg29ZS/OTLqfz6YWSe9pyajJ6AdC4AClAutuwHcts2E931txF7yLxfw8p9L6Dc6/6+sfjJMzZHtw3MTR3ehelO4QYzEFnEL4I4ULtHpbcEQcfaxJnQWn5TEjCtJhBtrnGOLxVSB8hES2/tUUXdGN/MaSMASRtkZhq3swIVB75MDx2dMBMRtwHXJuMVBTgU2LIf5PMhhLfq0AikMvjnHIpdoTCwFrcuhduZxg35jRJbYACN9/koKPsVSVnc1mF0qBKv/bSKXzwfSWVpRUCEZqR3EalzXkaQ6aR61RehWoJsN5LvhY6ZiLR2uj54UagEyOyF+GfMJfjb9Yy87W/4v/gu/kNPxT//SdhDjh5dEgWqf/0F2r/5R52fJqI9eEwiJu9dHlWqzyz/Ywfpk3qxczK4gUbHjHhlkACwe7H01kZW4/5e2sYDQpL8CGCaPutoommzqk6g9CZjTFQYp7gUCBWsYHpT9To+TsEIMdNz4PbaR2X0jn5puz9i4b4Vz7mHA99G3SmEDm/FMuyxixFTIPr3OqJNgg53o1UDFrAR4rsptz/F9ZcApYU4A8KzYNMkkGQrqujONfUmAhdfdWAjBCVUFdOzoJ79YFPtZDuNAJUQSVu8I5YT3rMeHdmJmb8MIgMmA1oFF4HJAhBe9xvMnM6vwjD7TUAauF/bWelnUnaULt9K5vQZeIfkG9001RxiK2CrgIIzk1R6k3HeVcZTYtDEJWOP1xm/9BCAiYpzfEQKnkHyPjpUnZ4DN+kNCQLadl12HvuI8lfPfS7K1wFDFCEdHUhvF+Gtq4hWbYYASKUQX5Csm3p/6BsBY9BSgBYDxLdtp4qI9RE/wyiMJfjj/xDe+HOIArQ8DNUiGkX1z1ML9bON7jk/O47y2jQKkLX13yW3YQ2kMtgjTwJXgrAMxgeEaNXdhHdcvab3T8PXA/tPQMCvtBxV7aJsSiuO8hVbSQ314B/dDSmLjmRAPcRUQKNkxM7eKr0xhnx0nHfz5vlfTps3su5eh5wyblQnYCFlNPGwUvUFIgWR6Tlwk2RHBaR96bd3H8nna8ALiMHz0GqV4N+3xi3NacjK1CwvKUjKEnd2otUIggis0IyiiBHAId0LGEUUIPmZ6OAGyM1AOmeBWARpmkGngGMcTBMpkvaQjjxaMoy85rnk3r6azHPeCza5r1T/+EMIhr8KsF8FlH3lNaXS58+6TKvuEm9ZDq04Kv8cINoZkD6+F9Pj44bj8eymCoSAArLHpbeJt0rvzs1fx9NYMNGutz0rvbX7rBHEGjSIwBysm0hlSnwrpYG0+qqLSaT8tXOPQLkMOJZmXITkMm22LkwRBCTjo8MVopXbMfO7wTOoc4j1WrvfQoUOH+nOE972G8zse9D+tbidq3Hb7oXyEGbRERCGkLgGVU1GBhlBkNGHqymgiqoSd5aC6rSAnELKgAXx05hD5lP8yKWYhYeSuuBZ1HGO4B8/xSyb/3Wu2LjfBJSgfJpQL0EE75AchBCtKVIeDPCP6cZblofAoKU0iAcEQDTx0huTWXpj12UyTWZfjQudyOv76rPT3XAik9TYIDRjJ1E+Twe+2/IzRAHXZm11CklIBMl4uG0jRCt34IYqmDmdyZ6fNqmmnd2JmdFHcM030MIQVCooFvHyUAvUQ8MIIoXIoU7BxUJBEglZAWvAM0gt8C3iN55jDWItGAEH6hw4F4uJBw+qYA3iG7QQIvkOzJxeRt7+XLp/dAJ26dFU//hjotW3/rnv72wG9r+Asq+65t+lz511i1bdcZIx2CUZ1DVahCs37CTaWiZ1dBem28MNA5EBCZuzobFLbzr5pbeEvVF608ktvbVP2B48yJT7wd2TJJ+PAG9ucYo4EG3OeKbiqZ4N+WwZJrx3O+JZTGcaV6hirEGMaf81gyWiHeuhEqFBBlwGjWLJ6PCujoNoyXRo7iUygphYSH4spJRtlJ9SFtIWSXmN96wBQEN3kGdKifTxDKAQRcjM2bCqn+IHXkjn1/5G9aqfY2Z1fBJG2P8CSngLwu+14jBdHnZ+mmhdGZM2RKuLlLdV8Y/qxFuSAzVoUUAtSASESZazv0tv2s5pCjrBLGjyS28JRiB0aC0QmRbL/usBL7MXKX/93DTwU5RHj8rIjaO1K4+EqZQFKUjOr4/ZCVfuQFIWPAMOdKCEA8j4EEQg0tJp6LYWUI1LzCIgSTaDjLvy0L66UQ5xsdBiOSUZUtoiGR/JepDzGo89C07RqjsoZSQiiCc4BQEIQ+yyw4hW383wqx+H9q/b3POLkT8AU0BASRb0h9Jnz7obOFwrDjsrBVUl2lxBOr3G2tAN/UQbSvhHdGJnptGyoBUBMYw+4crs/9Jb69dOaO0KdN+W3kTQclyGsHLQ7MOZ+khz0lnai/I5HPgNsJwkq0Y8B+JABVVpmdTQmgUJ+4dkyoEWq4SrdhBnHEkm4kD7i7FMhBaEhrBiUPYeAkl5jtE4hUqIKwTgSoCCNQ0h5VON6Eojvj34JCS0buVwDunsI7z+SrzjF3yICWDYd7wUAAWtKnZeCjPTR8uNyc6m0yPaWKb8tx1UbhpAq1E9W8KzqHooPuChAOqmRulNGMf33c+lN2sa5YlSuF/3AB38x0VI02NBaUII95J8HgXc2pCPglMQh/ghWAfIONJj2f9yNwIC0doBqEYNmbjWGx2RTq1s2MSZlmkIUDrSmK40kvHBKVoKoBIefGuqIuAbWqfXKDgwC+Yj3d5fmAAe+4jsq6+5spYFXQ2cTaQogl2YBgduZ4BkTD0bIlCCu0eINpXxluawi3KYLosrxeetiAPiUAVkP5Xexuky3Y+lNysQONxglToy+VnMNNJepsrQXpDPK4DPJ8eQgPhRPVQNRDQQWtd/REDj2N8okPHqUw7czmLrjDcASQyDTql1kLjs5mNqQdZvyDMVNy4YA6FDgwgCAOWAxjPgAYHidpTq9xPxDS0YoOI6pmoGBHDJqBbKCMzCNKa3kQmhgC+Ybg+tOqq3DlK9dgfBPSPgtC4i8S2IF7vT0ED3S+mtFdl/pbeEZKGw4nD95aYJCJMtFnnwrSPJuLO9gT2Tz3kfA/k8oqARqCLZqB4IiGoiHkBM8jg5P0uYElgD1ZBoyzBiTfuETB1E4RTKkpPSHIHD7Szj1g4S3rOD8I5thLfX4q7tRBuG0JEArDRkZc2B5yEB0haqEbqhgFs1iFs9BJUIPKEFBdNji1MvA0qyoHW1LOgDwDsRklExi9MgcSaUNmBAUqb+2BUj3G2DROtL2IVZ7LwMpsOi1XiNCE1Kcuh+Kr0ByP4vvWky+doVAnQ4QJLXpufA7WNLSbOdhK3sJuVvnPdDVJ4mEoFTEJCOALGKhiYRoSbSwZlGqDDlWt9SFrdxEB2pjs5+REAVxICLcP3bMT0zwVqIIloRRBTVfTkMXdFCAE5bBp+7+CRRBMS3jSypK42ZmauLSCvhlBdRIllwG0ZwG0ca0sl4SMaCAI7RGEApu43RxqkpoERC76pJ6Nmgi2MJAYJdnAELblssIQtA43Esouj2Icy6InZuuj5XznT78QK7g8hBchLYpJbeWpF9XXprRQSsQDXCjYS4coSxgBEOEqbaHLj2CAhCC8pGJkjlm+d5qvwF5BwxEYQKvsN0hyAK1fgGLAJWG49DC5FBIxsLyU0t+RhpLOLvKIJnaMEYqJZxm9aTedorEc9S/vk3kJqIpgxWwEi7ptgGTtGhClF/qd6tZxd2YWbn0aqDwIEwdTECIxUohdjFXZAy6M5yQ7xG2pxsLiVXcYWpKqAE5THALSTlOBQw8zOINbitFVABX0ATEUnGoFVHeG+h0cY9M4Wdncb0pjB5D40UrThwDtDJLb0pAPuv601pYAVEoOrQ4SquGAKCtKbIB/QcOBE5GHa1rp2gfGaoylUoR4sNIQDJOkxfNW71tWBAxIEYNDD11zS0NFAQphziW6JtI2ih2jrhWgxaKeG2rCf7lJeQfs5HKH/5FWi5iHCAYSTu0rNQieplOlsMMYu7ADO1JRQ6SHvYI/tQB7plBEJHOzRQpMsMe4f7VSaAYT+Qfc01twLvbBmrETjMnBRmYQYUqLamuOLHzQpCvWuuevMg1Rv7qd49jNtZRSxIzkfSPhgLYgAOrtKbSLww2BCP668QbS/jhoMkE2rP/pkDN802YN0E5LMElZuAo8ULIVKkJ8SbX0GsggrGd+ApLjS4oocrpdDAgmg9piyquMEyCE0IuBC3dT3ZJ72I9PM+BDtuJbzjn0hHNwc0aYukvfraULSqH0zSUTYlESDnN+4tt2wlWj2EliOw0t4kka7Nv+MOndICSiT09/8B/jZKQg606pAeH7M4A75A2dGCAlaQnEWyFleICFYWqNw8QOWWwdrjYVx/FYTGZ3Ie4sUyQg7M0psR8GLxuEYN2u2oENXCjQSAgm9A2FOmkUlZf7q7dhidG6d8jsLJTQgLjVdFohA7O8BfUEKNIh6Ir7iqaZzjM5xGqx6IglGmMuIZXLGKjlRaJ1wbQ7RpDemzH0v6Be+CYBu6bSNaLoHn75Hw9jsKmEZDgttaxK0ZiMf8CFMOBdJevXzo7u2HcoTkPLDSVphiBGCECeKxH1HlYmAT0DlKLhVXF4ddnMFtrqIjIfimrX0lZWoBKLiBALejSpgqYjp8TJeH6fSQvIfJWrACDgiTuVCgD1x6k/1QejMCBkQEdQqhoqX4qO2Kg9AlG+A8w15BHnyDSEX2maRuGad8TsbJVRiy4pUgUuxiMLMLUDVIYHBFSzTooSUfVUEEMMoBgWfq8tFqhKS90aW3nVuwCw8l+/y3Q3EIgjKgzSn2+BsRjIGgihsawHTPAGOmgozimXcFJG2R+V1QclNMPhaKAW7VAHUyduwtjhXFLrBDB5SAcq/9e6H46TPPB/5JMxUFX+pt2rrD4HaGECmkhLaYuGkhxg2HuP64FTJjMB2NbMjUw0LaNFJjAVQhUrQWKKAOHDGTWHqLZYM0ruJA1UG8nqVVF0cEIagqYuNsCNlfYplus544/xyHfE4hkuuwiKSGEKfYoxci84qwrYTb6RHtTOOGfdQZxIIYBeXAwSlaDFr37boILRXIvfDFMHMhrL8buvpADKhjHDSNH5KGfIYHSJ/1SKJ1K4m2b0H8FPsdAUl5uE3DmLSH9GahHIJMkeYKVaJ1QxBpQ0bK2FjQCqsPHAElEvpXTUIvBL7WMsU3ACzIjBQma3HbqmjFIeMpNUkjMyJFg1CJdgSwrYp4Un9PMhbJNq6kbeM134AviLGjf6kdJGPa224uj6+jMqqk8idCAnEGBkQODeMsJ3C1iOpXwoYUMbGkPIOMZRORabFMTa6mPVS+VZNPYK7DQ0xmO5JOY44/GclW0XvvI9qSxQ34aOSBBfGUAw5j0GqIloLRWbsIbsdmvIeciXfGxbBtLRgL1TLSN7uevYT9dyOZHKi2kU4r0fq15J73GtJPeSVDr3lcXUjEApoaN3rBbRjCZj1IWaju56YEBVIW3TSCDleRrAfKuJCUoIGb4hlQewl9vSahI4A3NkskvkHXRWEXZOoZjQ6G1PGEcWOTzjARIFLcUACDCkjjfV/Ab0goeRzf+I2ArUcsFIml0mbStdIIpzgHqIKLM6xGJNJxCqFrvA+ICMQ/A19AAGS6XfrAZWVt/WcVbaj833mnaNVcJ6JiewaRzl7MISeglUGiW+7E7YxwUQawYBVEORARK7hiFJ9JZUav0YRV0g97DGQ7YccW4lZsmL8M79jTCG78J/TNGV8JzXpE6+7DP+lM0s/+KOy4AyoVEPYfAmjTGrHf6I5zG4cxy3rBCjjdr+VRymF92oH4ZmLiChU7xysceAJKJPSm4qfOXAQ8jV0RKIhgZ6RwWYsOBGjRgW0nIhlbSLbp5q4KZYcrRqM73wzJmSFG4gAkloWQgIKANmVNAOpASJ5LbENsfPXkwE9g5EE8B649v20vn/Nr8pHrEMRbUADrINeDbryLaM0aXNCJaj6pCqAcsIg0MqBIk20CCjo8iF12VF007NycyAmBgR2kH3MJwb+uJLzzVuzSQ0GkVUTJoXLoUD+mq5f8K/8HELQwjKTSaHGYKYMm6y06WEG3F5HZeSjtx1Kcb9AdJbQUIhlvQnJVAR3RA6kE10rudX9/ek1Cc4Bz29aPnWIyBp2bQoci3FAIgcbZycQ7lFqn34IgzW3XyftOGwGg4Np8L5Gm141BDPVokaNMl74O8irhZbRC5dvnH60V+Qeo+EtHkEwEQQodXE00aNFoFhrapLtN2V2Sg90i3b+eDiJwmkgEQUcG8M59IsxeCBtXjy4jD/fD7AXk3/o5ip9/Z01E1yCZDKZ3FqTSoADaVM7bRu6Fb0aWnAoj9yI9M2sxC92yHsl1sp9pbXryTGOjakcK0hYCxz5HgDBChyqIMROWqfiCG3b9B66AEgmdV/zkmTcAJ7drANBQQRTptticQYcbu/81JM5S2LuYJklJqzuQsWT3IB1dM92gsKlWfvs7rfJZqlVuIHLWP3wE01eCso8rQTTQjY74ICDWocruI4L4Flcox6Xs9P4r8yho6FpnvaUy+MuPAReBtJbT2LoBmTWf/Lv+l+pff05w3V+IVt2FbtsEApLvrkUXGIMOD2CXrSD98KdCdUtDeN0zMD0zIKgwdVBQIZnbGNY748yibhBA90N7fCFAiwH4hgnhAAt2jo2YIIapyRnAnWPNFCVQMIL0+thZaUzegsZrK/rgnm0mU8UP0zL9Ck1UvnNBH4H7J5Uo5x+TwiwXCAzR5gzh2jxuyAfr6rHbKPHxyZZo+1AthpGUB8L+Q7URQgxotYyZMRezYBkUhhKjN0to+yYoDpN6zLPIv/3z5N/2ebIvfTepi/4LM28xrn8r0YbVRNu2kz7rYph5GAztBAeYPGbWPDQImBIoo1GFlIcOlOvlOFJ2/zRFFAMIFYSJC0jA9NrBg0JAudf/PQROSiTUHiKFwIEvmF4fMyOFyVlAYV+LaLrMNT0HrpUvNMnHJ4iu10Jlpn/6Qsyxc3Br8gR3dhNuyIOCpN2eiyLlgSrhxh24zQPY3jySSUGkUyoB1XIBM3sB0jcHykXaYixUK7DxfhgZxq44ltTjX0T2tR+j491fIf+OL5N+xNMQAXvIUcTmAXWAh124fGrNRdRdV+TcjkLc+Sr7vj2+FKLsBhZwlKOdbvMBL6BEQv8oxhL6z7gmCkSxcFKmnhGZWkjeUCdUcOw/RKaivKbnwE0+36uV37aRAFF0lVaD5f45D8EsmE145VaCWxyu6CO5CKzCHpbcSHloqUK0fju6fQgzswvTnUOr+3uviSACaPKcoNpYz8l2QBSNbS4x4BwM7IDN98O2TZDK4p3ySHKv/yxd//Ot+mZWRraASCMoYxcdiunshTBgaqCtB7v5FkYCtL/UeKzsG4ygoaKVCNkd8Uk9ytH6oHjQCCiRECcD16CMh0REviBdHqbXQzot2Pi9SCe3dCMcIMjk+3Wat45e9znvMsLodG/FIYCl+tubiNYOx3vR9kK2bgR8i/aP4DZsh0qAdGQwPXk0VPY7BjCGUTiH6ZsDfgpUx/4GIoCASPK9KiXYej9sX4894+HInEVQLACxsMrDmMXLMfOWTJ1OOAXQXQ4Xdv0lqEbgCfsEEQgiCB1YYcJEIL6omWfMwSOgREIu94Z/PAz4DTETEpEF6bCYHq9+lbQBjd/X6bvsdJlwL3tZFa0H30k98/L1JE0HH0Z5CpkUrr+f4Nqb0ChCujJ7vuisgDFQnyQ/SFQLjAEB6chCxocoYkrg2ZZtC5LOxjJR2pFkM4mEgOSxxF+/YytUKmAtSPzZUgm6F+MdejQ6PDDJFYnEjSK7vxdHB8vgWfYJBogcRAoiLW6ythEiYwjVwUEmoITcG//xWODrTBQHhAqAZC3SGYsoa8AALg6dFsvuMD0HTpLnmlxBX0Min2cDb0EkPiNmCMmkkHidZvdJ2ngxgtvcj+sfQXwLRsCaegaEU6YECuKbVgFo1Jr9IPHrLpEMAjLqmoQkMmreC6GqQAZ75ImgTArGguc1Qgx1rIVcthEioDpWNpQ0j+hgCSohWMNkI9agIeCUGFTB9+ODaysNpwO0zXEEJVR3cAookdALgXfvhi2SrMcpeAL1KdmxiHwBA+hUkdGULglOt1lrU0t+Kx9PX/LX/lg+xwP/R0J8lxKAvSMfieUzVGxITQTCCMmm66IjjJgSOAcpD4knuiO0Bwc207iqgpg2WY8AjH4MozezGgHXj3fcaZj5i9HiCHsFSZKsoALDg8K2LcKGtYaVqwx33GW45U7Dhi1CNg3pFDglQeNoLcOh5RAdqoBnmFSUpDPRJfLxUoqpWoKtadx2gzdoKAwKYdQqIQQ0VB8MB7WAAHJv+sf7gWexJ7hYRgKkBMlKLQySFhIZJTGpyAHV7DCNtpmAkaDApY2Ot/PzwBVMFjaWz5Z+3EgZSfvEgFIXEMYwVdBIkZRF0nbsrCwsQedi6D0KwgK4sFlCrY8TGY3OguLNrmb+sXhHnVKfuo3IXhmtE1WF8rBQGDQMDQgjQ0K5CGHRUB6ybFxvuPY/hutuNqhSF5Fqq4HiS1MWVIZqCFYmtTkq2jgCpQA8AcDzlLzz+fc9jnfcvIkPbxzmz9sK+IUIr2KJlFYcnabXzKMVXGH9611xy4IDXkCJhK79LnAKsHGP7e/iEMAXJCNI2iApQTxJFK2A7ic/TM+Bay9riSOZ1Qde09XIXhO9WBmrU+g76Wf9dYQGfwS6J+3mYQ1u+xBuuIykvNEjpTzbEFLkmDKoIr6FTPP/rl2V5VxDOrm5MOM4wDVEJNIsojalt0ZofMVFQA7/lAv2zh+WDlwghFWIIkCUvrzh8L4UC7p80hmDn4O+bmFpp8eWdR433C4YA9aOcy2oEjSyIGuZFAQw4AYquG1F8AQE8mnD2i3CW25fwy8Lw2xeuJTfz13Bd4aVylCFFAYxJBjQikKgZ5CAlrblXXHjj9DwE8Cyg0NAiYT+CRwD/IWYvSYjACsNIaUMeAYsYPb9lIPpOXCJYOIp5vHx7PE1PicKNNkTVnEkVwVVsNL4fNYiGQveBKWkNMRjGIsPx9nP+4EzmSx8iw6M4Gohvh0t5Cgu4Kc8cI4phQgmmxp9DlC1RHwXh8T0EBapk18As0+G7ByIyvVoaUxoLb2hQiImYyHcinfqRdjlR6MD29kjNBkjmfVhYU+acgRfX9XPO27bwLvvXMt771rNe++4m4/ecyfbGaCyPcvqDUIuC6otVZbWx8agQ2UIosnZFyQkpdAIcGAMiPP56ZoBbibgIbk0vYP9LMj6XN+3iG9trpCOHLZ18gvRVve+wafKkvIvP5xV1VeoC+5D3VMRD+DwtqN4DmAJ9QMXFT96+oeAtzJudGIfMYA2rSIqoNNi2asIybRxY0YfhREqrtoQSnxUBRokx1gQKRqXVlu/ZywvvyEu4lNyTT4WUdW1b1NWAJJs2PFA3F1b+7mzJp8TgXdO5sh8ihXczhFkV+tJ2ih1YQ0EEVOKyCF5H1JxGc4Y3M6tEFZHr9QbC65KDHh56DsaKv1Q3Ny4GgM2m9TDpJ18DCBocQDpOozUWY+h+JWPYHtng4vYLQQcSk/W4lmfH9w+wBfv3M49pSo+1KPH9zj62KOgq4Nv3nwbTy46FvTPpDq/hDGCKgnJMS6JhGzcEVeoID05qIZMGrHgfAtRBW7qL9ADbB8uMFKT6Owt28gvXsrqVAdDlQo5P02IEoOkBd2py9xGXQn+INCHCKiljrDwIBNQQu7N176t+JHT/w58E5g5cfe0RzV5MOaIdWl6LACtV/RBObqmdeirGX2khcYZjAsUqiFadVCNRVONpVMLnCYikDjarctoI5xTUJJsKGWIOjxMj4edkUZ8QQMFoZlEPspYvJsGf5rUdZ/IEe0YTm4c2uoo41sEQZlaaOSQrI/JpYgGy+CncP3boFgAayF0IALGh7AM6hKxAKR761EXUGkzVIbAeGAzSUoSi6ylPKcGtIB/0VOxv/4mOjKI5DrYHRzQlbVY8Xjr1Zv44up+ZgGHegaAQuhYPmsGxy5dzPLDDuGG2TP4+U9/zzHbu1k4YsnkHZUq7UFRBBHQ4QrSlUkEvRdJskgFhJQRdlZCtlUD8oBYw0joKG/biQ6VOGtBNx0dHa3zUhXUlZGePusdc2YfAGLB2IbkxfYdtAICyL3l2t/UJHQ48DXgibRDJlEACjQLSwREQYDkhkvyXhsj6QG6QVaok0gmfiyJx4kUyooLHBoo1AWjUI2fhw6NRksmLn8lp8FOEEFaOiLd9ipua+WmKFP6l7c8v9nOTP+3Bu7QUYcJeoAFHGOSvuTyyyrfPf/dKH2TKSC3cwgtB/G6j4Lx0IHN4KWQfB8i0f5vSFGSxfNIQZpm1HWmYaCEZHK4HZvQ/q31gaNEYSwRD6IKhAXwO4lpFVF5OxQ2giuDzSZrPl4OtNp6IF5hC2buCaQufCqlb38eu6J7t/ZIpS3YjMfb/7SJT90/wLFGSBvB0cAHRoaGWL1yNcMjBY6sSeiauTP53dqNnH3sEmxXufk+0X7fVDFoSKgnC5VwEhpqEq8ZIwwHUS1CfECAlGdQVe6olHiCpPE7+igOVFpEptu34j/sPOwhJ0M0gtg02CxUhwDTe/AKKJHQTuBJhQ+f9hKUTwMZxotOQnbiADRpdBFt+qtdEAEVENMkqERYgLTJonS0rCT5CLo3ZRIHAqb1dZHEqqoKLi6J1Utb1MJBoHHZTEc/d5r41xCLRhBfmCQKCL/Gyh/EylWdn/nP/SS8q3Yy6ZtQ3gdk8GLZqTIOvlL57gWLQS9lsrDxwvRwGfEsdWwKt+kevBVnIJ2zCO/6W3zzZf/iGbRQASNILg1BlEgodHUBSdqiQQrt347bth675DAojiRNBq4C1aFYQG3IzIR0D4ysgcpAXJIzSGYGWt0JYQnEJmmyc0CF9BNeTPVPP0SH+pGO7gllFgJk8z7f+89OvhjLJ2UER4K1wsBIifzmrXipFCODw8zo7eLGzdsZKDnmSzshtDyOb+4FJONByoNqBLKXmkKsAd+gThEAhGKklGthaSBAwSkzgMcc0tOmrCtoqYp/2gWAgaACmb7kEDSR7EEuoIT8W6/738KHTvsD8HngMW2EM/lI+2YHVEEdijQeStOCO0oiJAFD8l4isPhKi9ig+TWaEJAx2lE0sR0O0Ka1lvgaxcLRiGQ9phZEsWR0V9lMsjazj7gK+Dbwq5p0ttOG9HOv+FhNQl/FmO+IL49h/PwD+OzkZJQCziHWEPWX0DBCfA+sh9t6H3bRsWRe8UuqP34jWh5C8lk0cij7D/FMvTtPywHeYXNQ58ApABo6JJtCOjP1G6uWhojW3os9/VGggEgcHlQGIb9grB8GnYeCWQ+lrWCzDQlHZagOg03WyRCLFjYhs48l9aQXU/r8B7GH9YJGE6qCosKP7hokBHLWEDQJzIgQiLJ1Rz/pTJrts2bQl0pzJ7BmpMICYwHXJJ82EvIEAofbNIxZ2A2+gcDtrSy1ITatEBM3jpJsD1LlHoUPHDaD4xbkGB6s0IyWiph5ffhnx7dbvyMWfkhcCik8aAQEkH/bdWuAx9ZEdAnwMWDulFh/keSKymgZuqQ8pDhwrdkZAhqLB22VjwCKgDRnLuOUpcYRWxFVVAGnMOqaPFbV0V6TRJxY2Z/DRIeB7wBfr0nnP4xFIqGB6g8vzKCMlw3AecDj2duIQYtlJOWjQYgWK4hnGq8Pb0c8j+yrfgfiE/znp5jOWYBANdy/HXAKeBa3c5Bo4yB2QTcaOKhGxGB6cridRfAzhPfdSqowBH4KnAMEbBqqAw2R2Axjkl8I6qC8A8QgqW60tLWpPCCggBsh/aTXElz+U6KNazFzFo67FBc5AMeLHtrHNVdv5P4gYpFnWvqSPGsohI6tW7fTN3MbJnIUgfWlKkgHIg7VRDQt5wTRfHx3iNs8jFnQBVb2zmRzBcn6YAUUcEpXypD3DcVqhFPltkh59txO3njKbKrFgFDBkIAY3JYNpB/zFMycI0FDsGkaOBADYvoPbgG1F9F3axL6Bcp7gTcwlZHk2n7EC6BJ4AB18XNQt5ezPWn/vw0r8UNhirEK+BLwzZp4djBBqj+48GcoFzJ+AuBRk1XG0pESZn4O1z/SyH48C+rQ/rVkXvA9SHcR/usHaGEnkpvZeK9SbUgo5e2/OXC+Bc8QbRpEqyF2TlfjZhdE9efSmcZ0ZXCVLsJVt6Fr7kEOPQYGtift1VGlkdV0LGY8SO1zGpUhLICXB7+jITDxk9KeCFrcjnQsJfviSxl+09MhqIDxAR2XW0dGQh57VBc/NMK7r9/GHaUqWWAGkLOCFUEAzxMGqiFbN27G+R5VYGcYADKOdV9FVRBJWu8pVNGtBWR2Bzi3R00JSTZqkZSFqqMaKTOylplZn9XVgLWR8t/zuvjK+QvARRQqEaa58UUVFNIXP7219KOA8QA5cAVU/PgZJwKmNornn7spoRHgjYUPnvYt4H3AEw+etmgFkUQyhgcz/wQ+W5POd0mYqHxethu/H0uZFAQ3VARrGo+LVcQYMB5u4x14J/5XLf4boP4cp8l8mCDAFcvYbBcaRftp06mphQUEt20EHSxj+nKYnizkUnWRmpl53GAZt+V+gtuvI3XUWaCxgATwslDc2ijDiWU8SHZuQ1xeHsn0oSPrAWlaS/XQ6ja8U59G+vG/pvLz72GXHwZROO5ZxyPDVR5+VCenL8jy0/uGuXpdgTu3l9lYDSmi+EAHkAE27ujHGAFgJIpaRIM2NyK0adH2LFrvHrRIbxaqEXuEUyTtITkPVypTjoSuDsvyHp+fDMLrFnfzyXMXQBjSXwgb8kmIx0BtxDv+BLzTH5fs4YJk/QcLyJYDeSPqNuA7xY+d8Ub2gPzbr7st/47rngScC/yZ/YbueXaiJExzBfC4mnhO2SP5/PDC5cAXmQpYU89itFRFspk4ownAWrQ0hGTypJ/0MWLQ7SshlQM0+fqhElr/GrN/2q0z8WZYp43Mx2m9HBeu3E60Zie6s4iZ3Ymd1YF4Oar/uRKGNkM2TyIMD1wFCusYN6nOunzqeB3gd4KGIE3jeqplICL78k9hly3BbbofrMd4kLgCNthfoTMFzz2pl288agE/etwivnruAt7zkJk8ZXEXR/RkMMZyd+S4PYgoAwEx2n4wKdr6EiRrQrqzgA7vpdNTBaQzBYCLZXdIT5pH9eZq8pkPQZN8mstvA0UyT35Raz6DxjIyIHLfASugWuazFngN8LGahL5X/Njp89gD8u+4/qr8O69/eFy3/yXsRwnIHnTnTfNb4KKadM6vxa/Zc37HVMEIWqyC08aCfqkCzoExuB0rSV34OqR3CXVcFde/HknnRwmMIMQNFPaLgHCKpDwkl0Ijl7RfZ3xQcNsKhPdtI1q1HelOY+bPJ7zj34Q1CdEdr+UgIICXg+IWCEYYB8n6UYxkZ8XycYl8AIyHFjfX38+/9etQrda74jBmAlvahKGKY7C/SrEcsLjL8vDDO3jlGTP5zCPm8/1HL+K7j1nEl86cx/MP7UEQSmEIVtDxdORq+/MedHsByiF4BpTdRgOHdKSQrIeEjuJIwKOWdPDdc+ZDNaS/2EY+xuA2r8d/6OGkLn7eLuWE+CAAcvuBfhzDH4GPgT4DWFmT0PuLH90zEdUkdGUtngB0AS8GVk6ZvTVKe6b5OXBOTTqPqcVf2BOS7OejwAqmAiIQOrQSIJ5FIweVADyLDm/FzjsS/7zXEoMObkBHtoGfAWka1zNYREdK8RHd7FsUTGc62SwrcXgW8RTKBaJNw0QbhxqySqWo/PUnUB2CTD5JNcSCeDB4H7iACWNSSHoGREFrS6hYtLIVe/QFZF/3CdyGbVCtgAgTQYDQwVDZMTQUMDRQZaRQJWMcx8zyefJx3Xz0onlc/ohFPGpZJ5VCwCh0gjcCI6BaFzmRA0/2rAyX8epZkIaOSgSzMobutLCz7BL5NOMcrr9I9qWXgkm3Sa20nv2ITW0+cAWUTMJ+M3AXkAXeCayqSegbtQ2oF7Jn+EAJiHZfBk2bUHUShDPNz4AzatJ5Ui2uBthL8nkI8CamCsY0SmdB3EQQhI2wFh3YQOqC14GfA3UA6NCWxpgZm2oVmVAfWEoQgW/Yl2gQNU5kzaXQMCIGLZcxsxdglx6C7rwfHS6g5ajeiRbedDXhVT+H7qVJFhRLBI1g8J7dk1CqJynFNZ8tFDk0HCb9uNeTee7LiVaug2jPN/M6hWKoDBUjBgeqDA9UOHlhmtMWZCmWIkajE79vWNvIcncUQPZw6K5TpCeNeAYDlEJlOHDtB3F7HtHKlaQvfjj+2U+jLQpgrqENHgceFwHraJABnkctahK6X+Fq4AZUb0O5DxioNSAUiKk1IeRQekCXoxwJnAScBRwBsE+nJ2jiqjGZ5gfAx+NW6sngF0whREArIeoc4jXab1HQ4c3YpSfhnfq8UWmyDm6GahnypvWX1rNQDXDbBjHzeidjnMvYC9zdWRiugE8dwaED28i+8lN4t15H6bKP4bZvwMxdinT1UfrFl+k85SLondvoiDNCHS8LUQWtSUg6FteFMhEkMwstbQIXgvVHZUEERfAyZF/2BSiOULrs23iHLgHPB3XsKQIoMFSIYncIbdHWL0RIHiTPwbNQCtDBEtKVBadgpX6dCFqNGmW4rhTaX4GMbS9DY9FtWzDdKXJv+ypjI785iM4DunY98Ng2nUjPBj4PXAmsBdbVWrDvq8UtNfncB6yPX78K+DLwwib5TP7UBGX8TPNd4MSaeJ4xWfKp/ujCS4GlTCFUFQ1DxAg4RYMo3vezBf+cV4IY0KRnVws70ah92YiUhxsq4foLiG/ZZ0gjCzI9WSTrQ+gAgXSWcPXdRBvWkHrSpXRc+jPS5z0JCjvRkZ0E/7qb4pfeDl4aUunRU65tBtShQ6vQ4dXxupAyLsQgmdmxhKNkvL0AYtFSPwDZN/wfmac9l+i+NWi5AMay79HRFwW0jaWsQYcraLGKVsPkaO3dQPqyqACufXlYSyWiLQPkP/B/mJmLeWCkCvJbgIPpPKDfAG/jgTFAL3AocGx87QW1TBTVfbuWM803gaNr0nlWIp5Jkc9S4N1MJYxAGKFBcvQkQmPtZ/EJeCc+fXT7PaDFwVFCQmhBfIvbPowrVhDfA92H068zHqY3i4YRCIAiqRTRXdcBAfbwh5N97XfJv+sHZJ7+BjKPvoBo5b+Jrv815HtBDDFoLAtS3WB8tLQFLWyEYGh8IjJ+PRMCkkGnSD3EWChvA5Tc679J9pXvRDdtwm1dD9Zj0tHd/KwIIGihihaD+hUrTJhqhOlO14Nq1CY113qJMv/6t+Kf83TGRPiOGFMGONjOA/ow8KXJlIEq+45pa7m4BfqwmnSeX4s7mHx+PCn1M2MaIbJbX69hBJFLvl4MOrQJ//QXgPFBlVGmKW4H4/GAWAGUaMsg6hx4hn2Fhg7Tm0PScRYESLYDt2k1VDaj5Y3oyAbs4eeQfvpHyL3r+3R88NfIwiNhZACk9aA5XIBk5yJdhyLZWcmxDOPBppF0LCGNAIldLiAWKjuBkMyz30/+wz/GdPcQ3XMPWi2DsftGQjqOG5G2/OFSDy0FUA53q/tRBWRGZtfLDsYS3buSzFOfROZFH2J8yAdpD+YAP4bh5cB3xm0NmYLZyrS3SsAngUNr0nlFLe5lMkmyn+cCJ+3VzMUacC5pGlBloogIBA51mmQ/xX7M3CPxTr4kcXUCrloEMSC0R0E8i5YDou1DiDVNu9UFRJgU4tlvNQnF5UQg14HbsqYekukB59ChDY3SWrGIdM1BeufGsjVJgoCAsaAhOnQfhEUwKTA+E8KmkEwfILGEEtk3JDQMUQH/zP+i86s3knnK89GBrUSr7kVLhfhzwm6hCmGAlkbQkUG0XEzOJULbSGjMVu2kEQHFDVfAKRiZcBYk3RmkJ40mQ0/Berg19+E99Gjy7/sJ4+SHIrLq4BRQIqFnA9/RPRGJHiTCEQ4k+oEPAofUpPOGZCr15FO97KIs8BX2FsaAU3S4iBssxDGCGypO+Cx2RdEoQiDJfvrXNUpvqTxoa+GCoNrmF0BankrKww0UcEPF+mMUsKYhhiCcNAlpGGFm5JCM1xCSn8b1b8Wtvw/oYJQ9owAtD0NQBQHEtB7DbVIgoCNroDrAbmEaEhJjEwlBIqIogmCwLsPs679O56evJP24ZwGOaPV9uPUr0eF+CKqgTWNxVEEdhGFdNG7nFtz6VURrVtUzPy0MIJkMpncm4pl6E0a0diVaHBld7mv6lqgy5qmYxkAQocXKxAWkjTCzcogRiDT+9zwMaUPHR3823kzTAS9hDMxBcgzDs4FvPugqXMKByHrgbbF43lGLzex7vgr4e69lTeuy0UIlWYvRuCTiHAjjxykaRokIqkWkey7eac+hjigtRMH4xSGANfUsSCsh4lswUpeSVgKwZt9kQcZAUCFad1erLJE4QBFA46tJynAoiA/GR4ubobyV3UI8SPUlbd5oU5u2geoQuBL2qDPJvfnbdH7uGnJv/DT+Bf+FmTWvLku3dV1dLNG6Vbh1Dcm4LWvR4R1IJot3+PGkHv8Ccq/5CPlLf0jHJ/9Mx2evpOPzf6PjM7Xrh35J5pmvARxu9V2xdJk4mkhIiwFUdqMUV42gM43MiEf9iMVt2Ej2Wa/DzDuMcfICERliDDwOEvJvufb5hY+cvh1400E3TUAPCrHeFq/ZfbUmnYB9TZL9nAg8c+9OLKhANQDfJpJAAEWdIuPt4zcm/qs7FpAY3M61+Kc+C+leyGhk/NPNm36+eAathkTbB/EWzIRKFR0pIZnuyV8L6ss3JmBHEaRyuLV3AgPg+RBWAU3EjoI2rqIRKh4Y23SImwFr0coAuABJzwSbZkKIQKobwiJEZUDBmNHZkHPgCmA8zPzDSD/h/8dr0OKOeibkdmxAR/rRYgEUJJtH8t2YvrmY2YuRmQtpZxTpmoWZvwLvlEeRftxLKX3pzVQv/3V9PBDpzKg/YlRBpM35QQCSNKjgFDdSwfTYpDVbx182NXNyRINldLiAmdNF6kkvZ5z8SUS+BfDgEFAioTfXDqNbDXyRgwkB9IAVz1XAF2rS+TFTg++xN1FtyMdIqxsinVAGJEZwxaAhIWtBHVjBOzn2JZrIRfcoYUN8DzdcrrdnY0ArIcgkZ9VB1DiOuzdLtGkIyXfhNq5Ed6xDeheihf5ELkgicQwiEeJCEFDjgTJ63cbGe4RKm5BUN/i1EMOE8HJg/IaENAIxTVkX4CLQMnWsj+RmYA+rBeNEXftzuRDMwiPI/8+vMAteQ/n/PotdugiyedC2G2PbW8mauBRXRbL+qIx9TAIHWQ8zJ09w3f2kLnokZuYSxsEw8ATGiXcQHkb3pdq+n7vjDqc+DkaUA2EX62XAF2viuYr9TZL9vAA4fK+u/YQhGkbt54dFbtxmUKe4Uly3F0EHN2GXnYFZdtaktGuKZ4l2DiOeASMQOQRBmSQkLsX15pAdRdQJbtsGovX34M04GugHhNYsiFgEDqIqgkNNGpCmkS8+AFodhLCMeDmw6UYgjAvjNyKqgAsAWteGYohCIEzEgOxSKkCb94V2ZF/+GaRnFqXPvQuzaCGS7wB17Q+to01WZA1aCSF04BkknwYZRyYkQCVCZuaQjMMeehLj5AIRKTFOzEF6BtBfgaOAvzBZ6D6chK0cKGwDPhG3Uj9tisnHBz6/1ycWBFH7XecCxO+PhViLFspQCWKZCVrYgXfcE8de//N8QB+4EUFoxRPiv5DBmrpIdV+0ZHekMT0ZiECLQ7g1dya3Imm3FgQIKNKQUFRJhCAAMnp2XFRBqwP1624QiysD0iTENv9eWy2gTcIxIMl1PELMPOOd5F77Ydy69fGGWAOqgGuzYbV99UQjB6UQSkm2PiZOwTnMoiz+Q09lHLxYRP4J8OAVUCKhLbW4CHjrlG0AkINGPP8GXh43FrwxbqWeanwMyLA3UZLNldJmplu8NwNraItn0WqAGy6BtSBAtYDpW4x9yJMYEz/NbqFJ+7hALEvHpKIKQYTM6ABrwKaIVt8GFMF4zRIdvRdKfPDSqJdHjQ+yq0zCgM00utvyCyDVDQgTQyEKQEPAtL6ncaC0YpJW7uTK7tY30097C7lXvR93/zqIQrRaxg3uAGPHuYco8TgGtBgfVGiFMREgcEhfL+HKKxmDD4vIVxmDB+OJqB+pjeH5I/B54EwmG9UHy36eEPgh8K2acC5nHxL87BEvQnk94IG+zn/yn34zRvYzB3gNexMRNHIQRWBM+0U7p2iliqT9NvIxEDmi/mFQBWNAwA1swD/5GUjHbMbCZLvBRWOsHwqg7d83Bg3iUk28YD0ZSMoSbR4C32L6crjhznojgg6uQzpmNaY6CJCsBSXrFiZpQhiVRWiE+F0N2YgBhD1DwPpJo6S6NplNU6YmwmSQrmVCbqSf0uc/SeaSZ0FhJ8Hff4s55OjkAD0Zoxanyb9HLVQRkwZrxj7WWwTTPY/Kbz6LXXoi/hlPoxW+JiJvI2Y6A2o9iO6mWpwFvBoYnPSSnE7QVcqBxK3Am4Fl8aicfS2fVwBfAY4AlgO/Dn768EfzwHyBvY0IRHHGINI+s7WClipQSbrk6oggvgdhRLQjnlbtGZL7XIQ95jFjpMxKnZqk0JA9wgjxUdmTen6QAtKRwm0ZgnKIdHY1NqSuvQtsD0D7LMgFoC6RjyTlLY3KcblN2OtIktUgHohtCgMiTCbZF3+C9CMfTnjbP8hd+hv8s59AdO/tYGyb34s2VTojoNQlNO6NqmIxfQspf+2V6NZVNPEdEXkRwLSAxqB2GurngEOBz+3XLEU50OgHvgFcUBPOcbX4WC3WA+xj+XS1Wcf5TfCTh5/aJvs5DHgyexuJy29KcrhZtYAOrAVNsiJE4qO1C2gpAM+CtQC4kXJ9Pw5h1HhdqaOlAcy8o7CHXch4kK65YFOA7kHJVyByUK5O7s00cJjuLJJP4wZK4HtocYho9a2AAdWmbKIpC4qqAE0bUz3AoeVtUN4OGnEwkv/oH9D+TQT/+hO5d/8c/+RziFbekQhZAR3H1pO4RVuLAShjy1Md0jUTdY7i554NKDFfFpFnA0wLaPynoe6oHUL3auAo4NvT4hnzxNHnAUtrwnlBLf7KfkSVS2nP1TUJLaaVzzIZOIUoAiGeENyP6ZyDd+qL0KHNUB0BY0kmJTh0YKQRQwVc/zBuIC67WZPIQaT+9faw88BLj8siZtahmGwXRGGSFewORnClKoTR5ErIgenJghcLx8sQrbwZKIHnt8+CjIA6CEuAS7KPOhZMCo1KDRGFRQ4+hPyHfk/4j5/EQroS/4yHE91zx2hp63gPPYzQcjC+5akoxMxdTnjvvyj/3xvQ8sirRORlANMC2r3TUO+sxXOAY4AvA1USUN1fk7CV/cwVwCuBQ+ITR79ViyH2M9e/5+xevy/1am9JHq/DQ0OliRTw95qEfGKCHz98OfCISSm/OYfGA0MRqWc+3knPxb/wUvzz347uWIUW+8F4ILGEBLRURYsVNAjBs00bHgHnEOtjj7qY8WLmHoH0LkIrw2NkPkIro08qpRxAqTqph9dpGCEdaSTnQxghnb24tXeg/auRbDcNZNcdcWJAHVIdAReC8VpH9Gjcil1tbE49mPCOOZvMM98FYRWA/Af+SOr8xxHdd0eyNtaMxoGizdsIqg4q0fhKceqwi44muP4XFD/wqNUA0wLacxHdnn/X9S8DDgHeBdz3IBtEqsBfYumsqMnm/Fp8oRarmULMnZV/2/aNRfOX36yltDPAn5GGSGliIXAFCe8hZlIaEJwjPqcHu/Bk7EOfCYB32stIPfl/obAFHdwA1kvEZU0jjGFXaHkQmXcUdtmZ46+l2TRm6UnowEYwds86NVVxQ6VkTXuyskffIp2Z+LiGPNGmNY0ynHS1z/gkObsHQKpDEBRB7Oh/bmPB+GhYgcoABCMHVVlOZiwCL0UMuff8ktRFTyG6785kijoKOsbCtCoY0GoIgRtbQqpgLJLvwe3c+Juhlx169F4S0DQ1CW3Iv/v6D9RiBfB44HvA8EE6iHQD8H2S8tpFsXSmrHy/8ZPbX/zY51/Nxe/5Fy992z/BWmyHRyucWct8LqXBM5gsoiiRSmEb3gmXjBKDPf4S0s/6CWIMuvXu1sViaSO24a14y84A408oY/bPfBFYg45sB2N3fynIs+hICVeowGSeG+S0ISAvLqOVi0T33ZSI5oGyIEDj8ptUR2rRD+riAaWWGMR68XijItQ3pxbAhUycqT/TK/f2y0g/9jlE990FYQhiJtYoVQkg0nFIyIGYuoRq8ZfhN55o9rKApqlJ6Fcd77n+EmAp8Kz4Zr3lAB5EWgSuAi4FzgEW12TzzLi8tpYpzhK48GtXru0ergQcD/xkdT9//NkaZGFHuzbSd9ckdP2kZPgCqKKhQxC0OoL0LcUe9USaMYeeT/oll2MWHI/bcCNElUQOSJu/MAWz/NwJ/wKY+ceRueQr9YxLBzeB9XfDPklnmds+BJED34BO0lienF+LVP0x2U6ilTcCg+BnAHngLAhADBgPogpS2o5Uh+LX0oAhkZgPIhCVIRhuZESuAhqBKi2oa7znQnABuGo9kudhIzSMv4ebElLKvv5bpJ/+KqJV90ClBMaOb/+7CGgsoYkd4TAX+MU+uI1NM/LeU9LAOSjnAWejnABkdjkeRx/oDA8FfaA27Kav13GM4mn9PluAG4F/AtcCN9REs4MDlNM885O4kw0jsDZwnDgrzy++fyEEVcJS1LYNXlXZJaoTaKHX0WeuhA5XKIGC27kSe+xTSD328zwQwe/eQnjtl8HPI71LUBeX8DT5oVouIKkMmVdegWR6GBttMUt0++8of/dF6MggZu5h8c9woCSo7vp3zyUvaCXAdOew83vjzbRKW1THvxVBkzc15aFbhnDrB1EcGEf+PZdh5j8UHdk8ev+NuiZBKLEqUQWJZYBJoX4nkuoE44OLAEcDaR31Q7wOh7SmBypJizoCJsUDolFT1ibsD8rffTflb70fs2AJ0tmdZOtGABk9CUikEZA0xKR9gNYSnrbNpl7b+Yl/f2YSM6BpOt57Q6UWf+q49Ia31eJMYBFwPvDmOEP6DzCyjzOdEOV+4E8onwCeC5wQZzgX1+Ldtfj9AS6fPPB4YpzCQiNcs63AL3+yGmZl2aeIoFFyzAIuwC4bO2PxH/UR0pf8EDPzENyGm9DCjpZNk1rYhll4/Djlwy5vmvboR5F7w9XYpSfg1t4IYQDGTPjbie+hg0WiTQOItZOTCUWKdKQhZRE/jW7fQHTvjUAaVMeXBSEIgNh4hlsVKW2FwkYo7wAXABZSPZDqhlQWjCUZNBrLzYVJJoMA8TqdlwGTjuUDrjRMZd19jSPD7/pH7VqLVf8AWInYtUClaVLCPidzyfvIveqTuM1r0IHtYL1EKOiumhMSQTmFajghdwp8Wt9+8kMA9tEkhGk63nfDduCKesSMvOuUbmA5MB84ApgFLANmxNELdAPpONqhQAkoAzuA7cAAcD+wGViF6jrgns5P/2cTBzdPATwSMFYQp/z2mg0rH//MQ2Z4VnrCUNlnOEedoIJ0LcAsOZ3xYA6/mPRhFxH+7XOE//4u0ebbEL8D8jMQLwPVAnbJqXu6zROZeSjZmoQqP3oVwRVfQLrm12I2ROH4Jq0rIIBvcf1FCBUzrxvSHlRD9hphBNlGGU4Hq6BCeNf1+Oe8BKwF5xIRCW0mZQsiiqogKFgfnIGwDGERvDSa6qDwjz9Apo+O0y4Az2ciDN1wJQNXX0Hh9tsYWXk/4cAOls8fJN+j6NJetDyMd/h5L8289LLLETMT1RXAUuAY0OOBWsgc9iGpx78O6Z5F8RPPg7CCzFoEUdAmu0+EjgCRxhujLai2TXpzKYPt9EGE8o7iny+7+KiFT/39HdX9JKBpOt5/wyDw7zh+TRMjbzvZAHkgA+TqoS2neShQBIaBSufH/13kwc1zaMIppAExejFh9Fzg7ewLBHCKRlG892cnZtGpSOcCxo14eGe/Du+U5xHe9GOiu/+E23In2r8WcSFm8UQFJEBrqQQR0k/7HHbZKVR+/Cbcprswcw8nLl8l3mlrJElOUB0qodWwfiaM5DMQhBApCElpRxWUiWNNPQvS/jLSM5vozuvRoXuRzvmNLDExZPtJ2cTyQQBN1oYAXIRIGZOJ2PC+VxMMlOl5zBNIrziC9PxF2N4Z2HwOk+1AnSMqjhD2D1FZt5rCnbfX5HMdQ/+5gWBbEZPzMR1dOD9DRBbTlcF1zUQ6ZxHe8tuZYowC2+L4BzGqLgWcBFwEPCZ+POn4515CvncuxQ89Fbf+XszCFXGmp6gIQkLLLNvQgZFkACqgsXk68z50pChtK3LFPzfzz/sL/HtTeZaUoz8A5zMBhGmmmbrlt9ntmj9Wh25oC3TrNY9fytZS7ek+WAMyAkG8/iPUu9u8h70J75y3sifowDrcjlUNAS0/D8TseQ9+LCEAt+UuKt99CdFdVzck5GfARa3rQEoc2vrXcSUCFDOjE5mRR4yJJ4G7upwkl4JIGyHjXANSwLdQrBKt2gkI0Zb7yL3hq/hnPA8dXgPCxNaCcMTiIQZcALMW4XZsZPWLX8Tmn/0LvwfSczqRTKYWWUhlwDnCUoloqEA0PERUdUhnB96M2Yj1UKeoNvx7yJKQvgWWaPEMEAPw+vzbr/0U40BVj4rXNJ8OHMUk49beRuEDT6qfvWSWHgu4xDVNa0Mtk8V9A0ZwkaMjbTF9GbasG+aH/9rKNauKrBmEdCbHsgWzIKwyK+h/+yfv3fyhg2ENaJpp/ps25OGnAHLWL+8nbW9gnyCoc0kt3fjI3GPZU6RnEfbQczArLgAxe2+hMM52zJwjyL7hKlKPfANu20p0YBNYb9cOkzbSSFkwBrd1CLdmRz0rwrdILo0WqkQbBxHfIhNdKwpdXIbzIXKAEN74F0DBs4BMbC0IaT23x3iwdQ2mbwaH/vQPHPaVd+H3ZYhKVUwmj1ZCov5BwqERCAWvq4fMkkPIrlhBas5cxLOJnQEUnGsSqtNuxomI3FGL99fiaOBi4IdMImbxMXR84lq8484muu/meIKHQRkHgcOFjq6+NCbj86VfreLpX7ubr10/QuD1csZRyzjjyIUs7Mtx2PIljHTM+uC7j5p34sEgoGmmeRZtyMF3iCFt386+IekoCqv1SdVm9lFMSQRARmUNqSd/nMxLf4ikM7j1t018grMVSHtoOcCt66/FTnS4hJ3bhVZDwpXbUAVJe4wbVcQzxALC9M0jvOVqdOvNSHbW+PcFISDxFVo/Z33YugWGtzLrRW/h8B99n/xRhxP2D2DyeWxnJ7ajA5PLIH5y1IMgLT9NEZyjCe1hNxCRP9Tiv4FDgQ8Cm5gEpHMG+Q9eQfrRLyBafQdaHALr7bI7N3kALlK6On2KIxHP/urtfPiqLXR3z+TcoxexcGYHGEAEYy0aBRxy2HK2SMfvP3bCIv9AFtA00+W3JcCJ7JrttyXNH8iFv7kcuJfJxjk0dCDx/p+exUjvsim+MUySbAjwHvpksm/5O95xj8CtvxUtj4D12iYZCK1PfFsPHSoR3b+daPMgpiuLG6kQrtqGFuPjJ8YpN3UK+RR4BjJ5os3rCf75ByANuL2TBSng+2AEGCZ30hM5/Pe/ofPkhxJu2z7KVSrUkfaHl+IQiDQZfwM97AEisqoW7wAOAV4EXMskkH3V18i99BPozg24rWsbYuYBGg3yHlEAz/7yrfxk7SAXHbaYxXN7CVXxfJ90KoWf8muRwlrLzJ4OFhxx5Kw7txY+elAKaJrp8hvwS1p5F5OJEXCuESJQHUZmHMqURnZdkpOehWRe8Rsy//V+GN6C27YarNfmnBtt77WUB57BbR/B7RhpdLNVQsKalNyOYSRlEc+Cjl2Gk6wPGQ/CCMl1EVz/W3ADSKZzN7KgmFQKunpgxkKYuRz65qDFAsVrrmDr597Mmte/qb7eYzs7SL47rY/R0bLT0ctSMb3sBUSkXIuv1eIM4HTg08CtwGb2EqknvJ78e3+L6eghWn1z0nbOaBcJited5utXbOTXO0scBazeuBnPGPr6ekn5Xl1CqXSajs4O5s2fRz6bYeOGjdyzdeiVHzrxkPQBKaBppnFOLxHa8mOa8P/rTz8CtjJpCBpFSVtwWMXMPIyEKZ8FtSzk+xe/k+zrfo+dtRi35iZw0cTXoESQVCwvp43ymwjhhgHCdTvQKJaLyNiz4bI+hA7TO4fwrhsIb70cvBljZ0HWIukM0t0LPQuQ3kOhdynkskRbNlH82x/Y+Y2Ps/6Nr+G+57+QVS97BRs+/Al2/vLXVLdsxeZzICDjGqyo9VAlmbcmAHSzlxGR62rxulocB3yKvYj3kAvp+MQ/8E59DNGqW5KSXAK5jGV4S5nv/nszi4GUZ9g+UuTGO+6hXCozZ84s5sydxYy+HgDuuXclP/rpb7jqun8xUA29wXJwwQHXhj3NNKcZWYGRo5VdMgL8hV3zfuBzTAaqyaBHdeClkRnLmeK0T2I0ArHY5WeTe/u1lL//KoKrvol0z0e6ZkEY7t7PUcAaRAyuv4gWA+zsrsbxCyJo4CCKaEZEGgJCwFoIQoJrfoH3kCeDV8+MYtlloRaQBnwalKC6k2j9/VQ3rady/2qq/z9Wr6S8Zg3Bxk2Eg0OoeHhd3djuHmzfTJwqKGgU0RYBUUEhfgxaDwF1yfl40MleRlXnA68ATp2M05ylazb5d/yCys8+QuWyD6KD2zALl4MqqorpSvGnqzZzeyXgcCsokPcMOwtF/n7Dfzhs+VIy2Sz9/QNs29HPtkIJB3QbGHLQX6icD/zugMqAppnmmGMOe9HsubMoh45W+N11oYvYNV8CCux1BJxDa4EAUYBk+5DeQ9gddGQ74X3XE22+b39kQclzjRrh58k85xtkX/gNIMBtvLPtKZ+tr0nblyXjo9WIcO1Owvt34HYWIIwapbm0B74FIyCgTusCEt+Ac8jMhQQ3/B63/TYksxAEMBYtlHBr7ya4+U8M/fIrbP/iu9jy3lew/hXPYd3LnsP6V7+Eje9+G9u++iVGrrmGaGc/tqeX9CGHkFmyGNvTjfEsgiLtk8fWdaCmz6kCDnAOEIAu9j4OmAdcAGSYJNJPegv5/7kce9iJuJU3ooVBfN+DQLnyvgG8pATakJBviJzyn3tW8/eb7+COtRsZLJTotEKvZxBj8AQ2DxaPPuBKcNNM0z2z7xJrDI5d8gsSmstwEfDRyfCPRg4iBRE0KCKd85CeJewO0jET8VOUr/wGgx99AoNfeD4jP34/pb98neptVzTEFFb33eBajQDwzngeubddg3f4WURrbkUrRTAeiO6e+5SGbHwPN1QhXNdPeP/O2nWAaNsIWqiAU7AG8QyS8cAzUJdRDh0eovLjD94D0Zslv+SZVIbPL3/rBe8sfvgpa4rvfzalT7yKka9+iIEf/4CRG/5NODiI6egkvWQp6WWH4M+dW+9sE2uRdm6W5CLSfh0IVRpIkuxFCi4WGXQVPnCqv5dLcJtr8XzAAE+Pf/eVScAeelJNQleQee6HIayQ2XonWzcOc+PaIWbRMkoQzzN01qIrjkwtRAQlRqEUhMsOqBLcNNM8+6jlp0VRNG94aARLCwr8ngfmE8C79np5OXLEQFBAuheBTTERtDJCtOleou1r0fIwWhgguPs6ouF+FAMKJpNHOnqwPfOw85Zj5x+Ov+hIvCXHYOceuod1NwG09XkiIczsw8i+8XLsL95F5bcfRb0cZtZiCCN2G9MQUZ0wwvWHsLNQlw6+h/i2IZ7QgVPiE2Qxs5cSXPurVO7l3sdIuGL4zXM/5M+ef0lqOHxl53B0cmVYKZaEUkFwTrCAktB2mAKgbSuWCkjbGaZOBdVRa0CdQAfQPwlrQQr8iFqo6hLgMcBTgbPZy6Rq2ZB32hPgr5/ipu/+ijVhxAIjTAQBRhprRrkDKgOaZppMNv2sahBSHClghWb+fl3oBgBonwUVgM9O1vidOkEJ6V3KmFSLuFVXU738Ewx94ZkMfuBCBj/yWIY+ewnDX3oh5b//GNMzC2/x0XgLDsObfximZy4ghFtWUbrmpwx/5930f+yZ7Hjnw9nxrkcw+PU3UrrqB4Tr75qcufcaxJ1S7yf3+l9jZi9G+7eA6t75edYkJTgRqIboUBm3bQQdLI0u53kGUvOXjrx5wetJoPOjm13+0ju+nfvkPacwL/uEzAL/772zlL6+iExWCYJkmUlog0ysainIqNecAho/EAAyQCeTjIisqcUXanEOcATweuA69iJm/uFwyZe559CLKQBmnAISgWrk2Bk6FmZSAycvnfPsA0pA00yTyWWfVpMP5dBhjdDE7xgfH9j70681ERCK9LQXUHTHb6n+6k2Uv/wIyt96GuGf31t77feE2zchuW7snEOw84/A9M0HMYhL7pYYi6SymM4ZeHOX4i08Ejt7KSAE9/2H4V98jp2feC7b3/7wegx9422U//FzdGDj7q8FQQDsAG4F+Qsu/D7oh+yRD39d7jU/fbrp7fyVlgoQhiACsjesJ8kJsb5F0n79ihAjsfQcGvS8hTbk33fbL3OfuPss5mWfmJvn3TpjZlQXkfWgWm09pVqSa+s/ioy+KrteBxIBdY0g0uQTkGMfIiJ31+JTtTgdOBp4K3ADe4l71m4nDei4RiQqA4FDgIfO6fnOww6Zc9h7blp95QFTgptmmksOW3JhJpudsWHdRiIgJUITf2Ac+E/5U3/1u6d+SVKdL0MsqNt75Td1kMrvqgFhNfDB6mUvflRw3TeeWP9Mx6zGJGprycxylLeFuAiEJqR9zQhRQJB0FpvKYnrnoc6hlTLVe/5D6Z9/xetI0XXkfOwhp1TSz/zSxyXbfRpwAQ1GYrmsQ+hHuR/YBmwE1gH9IOtAB8V4JXbNj4ofP/31bvvQJ7RcQdKpB94fpOw1JGVxQ+XZhQ+d9F/5t/3rJ7Qh//7bfwH8ovDOo9/auaP0oXQmYmjYUhgEomQpS9vX3ZDEe+xqp5EmZTlUIemMJIYc+wkRuQOoBR9R1WOAJwNPicU0cVRZ/Z/r6BhDPIFzlB34Ait6O64+YfHMD7z35vv/zJaBA2sa9jTTpHPZ56sqI8MFWmHrdaG7kXFi5p36P27HXS8jqoBNgSoTRiSefhCBoUFURXIzkL6lxOwAPiAinwYovGfeY2XW4UimA9SBgjrFWCHVaSnvDMHIBO/bQvMphpLOYP25SOdc0r0C1e2EN152ZfZFP3gngKouA+kAXStiBtlDcm+89pOl/734ymjj5m/r0PDRYix4HqiOTzoC6HjeU0Ba3nPbCu8HfsIY5D9w+4eLH3vIL/31xW/O8MJTs1lD/w6hWgLPB8zY60AJCiK7ElaSAQUOVIlJMQUQkduAWnCpqp4FPAv474mUCLeuXcO2jVvJNknJOaWqUAEMkDPCoq7M9Qt6Oj76tbXbfvar/pED70C6aaZ52tL5qUwu96RSsUSxUMSjhb8wAbwLPrvBdB/yXS0NgtNEKBNBaMjHKSAA8QieRfERDHwROLQunxgz5/CjENsiPBcpNmvwMgZ1tCITFCPJyQN+VpBcD94R5/xv8hFZXYtbE/nsOdmX/P4/HZfeeIydO+sLqEJlEjr1tE0WNFg9ovjxU85hHOTedPOduc/ce5qblfpSNuWYMy+isxfCAFyY+E3arve0rxgmTQigSDwRHBAAZjLFEJFravESYBnwXsbJ9o0bGAAioBC6ehQjxYnQ19tVOnrhnGtPmN/78fMOX3DR74dKp/1/+RywJ6JOM00qnXpGNp9LF4ZHKJUDrKGZvzJBzKxj3yjpHqiOQJSM0UFknNkPaBCN/ny13gHXDzxCRF5Ri0Fiyl9/VJrQ9Qmm7R/TNmdAAXTXNz1pcU2CgDRLLSNYE0C2azj76r/+nH1A7i3XvtKuWPokMul+LVfaZ5cyhlnH+5YFQofbVvoQE6Dj43e/XJdk3iAoM2aGzJinqMYd7k0nECS0WQdSHbV2pA4cglQj4vmAACuYoojIjlpcChwPrGYMBodGqAAzcpni0jkzbz5s8YLvn/rQY95y5mknXXDqiccv+sn6LWd8Z2P/mz565/q/EHPACmiaabId+bdbayiMFAgAY8yeC+iEV2wxfYd9QcMyKBBEEIQkLVKSRLN8jKDVIJl+EEOmBwrbfygif6IZExlJqZJKQRiBtg7dtCnBpGjNggRAaU+rhFTBzxpEHJLv+S77kOzL/vRzb/niw83cmb/WShWCMLHlHjcptFpA0h5uZ+n04scmdvRzxwfu/KQ5LPfysAxdHY65S5RUCoIKo5Em4bfbD5TsS0ZFIIwgiMAIwDFMcUTkJuDxtDIM3AT8AvhAd2/Xs85csfScE487etGvt2x/6E/XbnjmV2+67aOf+cc///qhv1y1gwQOaAFNM80zVyx+ajqfWxFWg3brP6uvC91qdgMz56FvNLnZoVaHQUxDKEGEBiHUgjCiZbZ+XF4iDKFJhJLKI9a7hV2hJUs25ZveTiSXBueS8h+COhArGF9AxxgyoA+cLqiC8QQv7cDPYPtmf4l9TOaFv9uWf/v1j7NLF74W30NL5d3zjYKRMdp9PUGrEW578cNMkPy77viSd0T25dWCkvEc85Y6cp1KtdLacEBLd5w27VJVRCByggMII7QSIVYAjufAYA3wbuAlwMXAQ4CFInJ8LZ5Yi3cdfeoZ3/3evfdf/enr/r2TScRjPzPNNB09PZ83xtYHHI6MFLC0cCW7iTn+ZeXw6re/iJV/+CYaC0UBFI1ca5+uAk6Tcp2SoA6MRbI9N7fd5BI5gxGkKw++hw4XIXKjRGaM7PE8Nw0VP2+xpopkZt2RftZ3b2U/kXv91Z8pf+WRV4ZrN/2fGxh5iPgeWAuqY/6zOAVDY+w/ngFV3EhIoaq7nKrg+iuPLH7q1BW5111/LxMg/947v1R45xHdlbvLH/JzMHcxbN8Eg9sE64OYpEOuLQKikAxGFwTFVeLsD46tTUPozr/z+sEpngUNAe9nCmDYj0wzzbOPPOTSTC47y0C9+aBUquIZmrmGPcA7+4PfsrOO+YOW+5tqLFKPRDyuEQoYoQUXgJ8r42duecBRyU4hjJBsCtOTB2PAKTHoGL0FyJiJQ1x+AzGK9M7+X/YzmRf/4eaOD9z8ULtg9mdwDq1Ud7m4b5tGzKU9IVWLj/xxCy//yTp+c9MgTgyds9PkfME5HeV3LYe47RNbC0o65O76sLc49YmgoLgAZs9X+uYoYZCMdItpvx+IOJGOIAwFEaAUxLLFA05mmqkvoGmmedqSeQs7+3reHYUh1jMURopUd73+83f2EFlw+uOlY/4OygMgpt0wsDh2NWRT0CjAZLpu9C76WIFdUwaKxBBGkPaRziztrRMHzbsdaY+C9QXrhZDuVNMz41tMEfJvu/a1dum8x5qO7HYtV0EVjY2T7/LJdqTI96bpmFmLTo90SvBTlt/cPsiX7hzgRb9Zx+O/vZof/GMHNu3RPSuDlSSZkrTFbS8+ufTZ05awG3R+4p43+ktS34sKjqAKM+Yqs+YpUQguAiEJ2jwWGgKqVsF4gisFaNWBEYDTmPpMC2iaaTpn9P3CWg/nHC5yjAyP0Aobrwvd3ewh9viXVe2hjzyHVCdaHQIxsXQmsmYhkMpfT3vCWEAJkUOsSTIq1XqItPORMopWIaGR4ucM1gTQ0ffz1FO+OsQUIve6q3/jLZ13uJ3V/TMNI3wXkfINX/7rVl78/TW87xfr+d7V27hpTYnQGWRJjndeNIc+YIkIt+0o8fI/beCJ/7eKa+4aJjcjTVenh4sHl2opJNpa/CC7Sccn7rnEzPaujkYcQQX6Zms9G4pCiBwxD7wOBFAqCWoNlCO0EiLWAJzBlGdaQNNMl97enOvMnxgGAcZaKuUKw8MjeLRwDXsJe/Jrb7dLz3s0YiEoggjjKHbFF0VsCkl3XUkbMs+/QoFKi1KcA5QYVJkY0vIALw3ip5Gu3s8xBcm85E878+/9z5O9RbNfbTyDL44f/2snX71vkC/fuIPX/3kjz/jOSp76rVV88gdrWTEny5MXdbJTlQWeYRlwxZYij/vx/bzlZ+sYKjq652TwDKhvcdtLzyh9/oy57Cb+Ednz7AzvPldwVCrQN0OZu1BJzhx84C5Ea6FYFIJQkDBCiwFYATiztg7kMTWZFtA00zx92YJju2b0fcSFEQC+79Xbr0vlAM8ITVzHXsSe9d7f2bknPAsxEJYAGd+4tKgKqXxkOmZezgNT2tUwU5SkqUHb605EAG3rIHVg0wZrq5DtXp953mVXMoXJvenvn4uWLjhOZuRv/sij57AA6AZWeIYscNO2Eh/42xae893VDFciOq0QRI5F82bx+COWcVxnnk/dNsA5X76LP940SHZGhu4uj3CkSrSl8BF2k+yrb3Xe4ZkzJWdKUnUMDgjWwsy5ilNAH3gdyLNQKVOLxtfpSBUUgK4pvA40LaBppumZNeMPYgQXOQCstQwPjVAFpFVA/2Av4z3yi981sx/6LDAQFEDM2FtRoiom23eFPfd9IzwwZZpxblcDxtpLT9p/RuPNp8ZTyHZ+kQOA3Csvv1Xe/O+HnvSIZV/4zLlzWA8Mho6sb5jjGY6sxVAl5PatRWykZFM+K45czoJDlnDiScfwjKOWMxQJT/j1Wt7zyw3ge/TOyhJuLjy79LnT5rKb5N5w21b/mOzZYoS05+oje0aGBD8FjLUOZCAMoVgCmzJooYpWQrAG4BymHtMCmmaa5x972JdS2ez8KAhBABHCIGBocAhDCyPAv5kE/Iu/+F274LT/wsug5UEQob19FMQiHbO/y9hUaKK5A061vXBQkLbpD4gVPD9Ash14y4//GgcQ8qK/vfLJLzz2mZ85e260HhgKHVbAATnPkK8FRqiEEQM7+qlWA8qVgNkL5vD4s07gjL4e3nfLTp75vftxCN0ZIdxU+MieTXW4/V+yIvMc3yqzZ0dUylCtgDHNWWrzOCclWQcSKIdoMUA8A3A2U45pAU0zXXpbnu3seGkUhokIfK/efj08VMAXmrn2utCFTBLehZ/8qbfiUWdKtndAS/0guuu0Iygh2Z7A9C35EWNToplkjEtiIJn46aIuojF6xwbQOev3/sUf3sYBhlzyl+8//6XHHfORCxfdvk1hOHQYIQa8uAR3xx33USmWyKTT9exYPY+TTziKJy6ay482FHjMt1ZRjYSOKHp24dOnzmUP6Lj0zm+7+emPpY1j7nwXd+O3z0QlWQeiXJJGO7Zz6HAFDAAPi09HnUJMC2ia6XE7zzWeRZ0jEZDP0OAwJadYKzRxFZOMPf2t//AOOf8YM+Owf2hlGKJK60ExURk7c/mH7BlvLU9YQKoQ6a6nX8qusx1pnQYNFkQULw0mnUK6+j7GAYo84093vfgpRx37qpMXfVuNUIl0lKsznqEQhNx+y52oOtKZdH2TcrFcrZfmnnLIIi7vr/DY762msr1KLow+wh7S+bG73xzNSP2lKx3RM0OpVlu8gzIaa6BShUpFGmW44SpaicBIxxRrx54W0DTT+KnUQ9QpCaDOMdA/gAIgNHE1+wB7xjs2pJ784zPNnIdcik1BZQSiADRCizuQ3iUbzMzDLmV8lEbLQ1FVEAHiUCaERmB9SVqvc70bMs/+wRUcwMhL/qjv+Oe65yztzDyv0Pw7EZfkto2UuPPWu0l5Hn7KJwxDioUSi5cv4WkrlnD1YMBjf7iayv3Dz9ZvnzWHPcQemnmkdnibZnVFZPMQVEeNt0seiyCAMRAGUC6DTRtcIUALVcS3AOczdZgW0DTTiJFDQCGZgE1heIT+/iFSQjMjwHXsQ1KP+9Z77fyTH2q6F/0cmwbnML3LbvQWnXqaPf2NjvFRJAYkKbnFaPxYxpgULSSIBfEaNznjK3TN+jwHCb8eKH3LwEnAdprIW8OGHQPce9d9ZDMZjLVEUVTfsDx/2SKedthSri5EPPHr9xGsKXyaPST32lsjWZ49x+aE2TMixEIUNq8DxQiIKC6CclkQIxA6dDAuw8EFTBmmBTTN9L6fDMgCEkinU2zdvJVSpHjW0MTl14UuYB/jPfwTN/tP+8WT7LwTjjBzH3Ju6hm/PcGe/d71jJ8yMUhLB9zuZT9pqYVBCDC5znjw6MHDHfBv4CHAvSRgBDJGWLlhK2tWrSGfyyLG4JxjeKTInKUL+e8jD+WKUsSr/vfup//+xUcs3vOjJm67V5dmnprPKTNnOsIQtM14pCiCXA4QGiU73+CGKmg5AiNnFP/nuD72P9MCmmaafHdXytZwkUNVyXd2sL0mnzXrtpA2Qiv8mP2I94hP3u0/5stX0cTEMyAFbT7KefQCt+xyoTv+vFNMqlF+M1Sha+aPU//1lUEOMq4L3UbgocC/iNG4KSEtcNfKdWxev5GOjhyIgHOMjBSZvWgeT69J6M/rizz/K3f9ij0hGVz6Y7cw/Zm+LkdPLwSVXf/9oA5SaaWzUwlDED9uxx6pAGJdqfII9j/TAppmmnQmjbFGatDV3cnQjp3ceus9uPgm00QA/IQDk0LT0c1JBiSAJv6Jn7ervSVt1xlBXYRkspjuWR8m5iCUUBE4Bbhh1NBVz2CBW+9cxY4t2+noyKMAToklxBk1Cc2Ah5wI32YvkP/I3a9lhvfPOX0hmRwEVTAyeh3ImMbr6bTi+8kfF66/BIGiI3IB+59pAU0zTWF4RK0xms1m2LhuEzfeeAdVp2R8gyrNfPy60FU4MCmTAOrGGq/T9j11cdt12uBKZaSj57b0Jd/5D8BBLCEFTmuWUNo3KHDLLXcz3D9Avi4hBW1IaH5NQiccdSgRPOsEeAN7g6WZC71uU543K0IshAGI6C7/O8UgaQ8dqhJtHkFL7iL2P9MCmmaa2266s3DnbXcN3nbT7dx2x32ETsnuWj7bgHdx4FIggRa7KqBtJyC04GcNoCAGO2vB+4l5kEjoP4mEIOsbqqrcdNOdlApFsrksqomE5i6czzGHL8PBx08ULmQPyb3xtiG3JPPoXKcyb44jiqQWAC2dcQkGUEW3jaBD4eLC+448gv3LtICmmebayLmt/cNbNu8YJGWFtBfLp5ULrwtddPAIqHmogrYe9QBIy8gd8NKCl5V69mM6e7ann/2DywAeRBI6E7g3kVCjPbsYRtx84+1E1YBMNkMsofo8wflLFnDUiiUEyh9PEhawuySnqf7VzUv/T09HxMzZjqAqKDEqGANiSLJ4ja3kG7Tq0MHoEex/pgU0zTQZz6ysBSKCsksursnnFg5sis1z4FqFpA9YekNBnWJzBrGCRhGmb+7HiHkQSagMnApsJUaBvGcYLFe5pZZN4xobVVW1HoVCkYXLFnHEIQtNRblyL60HvTPq9f81pzeid4ZSKScTljwPPNua6CKAAy3qFBHQtICmmeYads124GE1+fyBAxxVLTxguU3bpDxNkvK7DDZrcOUKku8O7JxFnwd4EEqoP5ZQdZSE4o2qt958J9YYUulUQ0KR1jerLjl0KcsXzll+LPyUvYBZlnmkZozOnx3S0aVUKqAKfkrxPOhMWfK+IQHECq7ozi3+z1Ep9jvTAppmmh+1abc+oiafazgIsEZGiIF26z/S1j0uVExaSPV4iAharWJnLvis/9jPFgEepBK6f1cDPjs8w+aBYe649S78eFqCokRRVB/ds/yI5Sya1fukY+CNe36sxG07dEn6v0RhyeKIbA6qFejIQ8oKW0YCioGjqydFZ9rGh+cJlFxWy3o200wLaJr9fiPZArwauDNus3547bWn1mIHBxineabn/wdN+Es6S0SOduhYRw8Z8PMGBLRaz34wfXPfx/TvzvXA02gib4V12/q55/a76xubPc8DIAxDKkHAkccezuzO3MeOg9MB9nBo6c/c3NT3TOBYtNQxe64yd7bHlatGOO8Xa3n8L9fyiWs2E0WOnt40KuCqig5Hj2RKMC2gaaYl9LlaHFWLp9TizwAHoHweCqwF1tce33CmZz44Hw4HqG4t3Z3qSaM6DgNJHDEaKjZrMLXQSCGoYGcv/kjqyV8eAmBaQpcB7yQBI0LOCqs37+C+O+8jm01jYwkFlSpO4ejjjiDve386HjLsIZ2fvueSMOtt9cKoLiHJGlYPVFkdOFYVQt59ywCP/PkaVm0o0D0jjQCu4B7FXmJaQNNMM80ngE4gD5wcwduWWXNXFj7y1W/fM0N609uM7KoVW5PHtK4NJdmPoKUipmtGwc4/5O0kwLSE/gf4PjEKWBEyRrhv/RZW3ruaXC6D9SxAvRSXzmU58shDO4BfsRfwV6QfHyGEJYVIWdadZj4wC3gIcNNwyKN+tY677x+pHyPuRqIjR9595EKmmRbQNNPsYfbTvatJx05gPrz5x1duvLFw/4hN5SyIQMoDY0atAQktxGf9GEzaoNUAUOyiw57pP+rjjmamJfRM4ObmkT0ZA/fcv5GVd6+qz41LpVIAFEeKzJw7iyXzZ190HDyXPST3zjuu8xalvuDKClXH0m6fmUAFcMARwCDwtN+tY1t/me6MjxvZ8yxoWkDTTDPNubSCAjmgVI66BwaCPnwDziFpH/E9cI62KIgRvLwFBS0VsPOW/SL9jG//knZMcy5QIkYVPGvIGLh3zUZuv+UuBKWjswNjLdVqwMJlC+lIeV8/CWayh3R84u5XmhnepupQyKLuFEs6PYZpEAKHCtyv8Jo/boBIyTguZg+ZFtA000xzKm0wgHNQqipYSbb7GBM/FkxKWvqzNVL8vGAzBlcYwXTN2OktPeYpPBDTWdAAcBHAKAkZQ8YKazZv59//vKU+9klUSaV8wiAE1CA8k72At8R/UlkVL+1x+Mw0I4AAYoXIsxwh8MvBgG/csBXfeBcU33eUxx4wLaBpppnmIbTBAiUchVIVPAMw6jwgdY0mA/EFDRWkIR/jC16nRctlsB7eoSdc5F/88ZCxmJbQ34G3koACRoQOzzBSqnD7XSu55cbbuPWmO+pZURA6PGuWshfIvfvO68wC/3tU4ZjZGQygJFjPsBD41F2DbFo90pm15iz2gGkBTTPNNEfRBitCERgqhMnoZKdo5BrPFUBIdXuIFVxVQYRUr4cQoZUK/qHHvzr1pC/9h/EyLaGPAJfThAJpz5DxhOFihY1bd1IsV/GsAPjsJeyi1IvIEBzTlWYOUElOwsUBs6xwP/DVa7eCk8f8v/buhEeO6zzD6HNvVXV198xwX8R9ExctlmRKlmUJdixFIqg1FqDElmUpggwZcJQ4SpwNcORAihMjNmAokePYCZw4khUkUuAEsGUHtDbLgjyiFlKkaFGiLHERSZMcbrP1VnVvGoMG1WjOtHo4PeTQfg9Q/2AwD75bX9/iBClAIlpAmAEsZATGQgk40pdAZCGw+EoKSQp13/cxoSGeFhF1BsRTQ2wIbmCQYMGqf4lv+e4DjJZ8DOhjWIZMaMmFlujd+wd7aZP8518tuBn27hVTYpbnA440VNBZwyLg0V8W2L2t71pOkAIkIssAywisMRSAAwcGIQ6gnOALpVp8DJh33/mYADKTAmxkcP1HsTPmP5W98/uf4URoCuoHttC6Htoo+NJr34gW53adNzVLL2A8x3gPk0PLDuDR5w+t8l+/YDEnQAESkcU0x0xI/vMnuym8sp84qOAqKVhDI+/ApTAUn2nzNuf+4MkrGAuJad07tNu8zJ9cOCdHDLiGu2a9gZnAY2/0Un6ncA0nQAESkYU0l8wK7bLuw8U///K3t0FnRBRZhmfwhV7s5Fk7orMuvYyx0NGoAWbRurdpM3P3xkcuXNm5fwEw4Dz1vIdZgWWTh6c3H72OE6AAicg8mlv3bOJ29sFXvvVKz1dfeqaHzNw8PvVgAExdfI5iuqbtq8Zndbjmy32MhcwAzqA1FeB1xsGcM/P/9r58yEGOZyykwP9tPnq5/9p5WdpMARJRgJ6gZj/82RcffushelOyk0K8p37ywUyauSNafsnq8OqvHmKsZD4Q0JrXuxPXy3g4I/7nDyzrpAQYaJyCmIPhpzsL2Z7dg1cwSgqQiMymuc3U+eHhwm3f/K/tz9lpMYEx4NzQ5GOnzX01s+qy1dH19++hHWQprXuFcWI+u+GtD67qWj8dKCeeRl2h4W3g8ddGvw2nAInITJrbToP7H999+daX+w7E+RK+UsBOX/BYuOi888O1XzlEu8gKWvcS4+jcFR1fXxVaDnvPcCLg8U29NzAKCpCIXnRHwAxG1j/cdtXr+PKPXi5/hNzs7uzyC/8qd9cT10XX/72jnWQlretmHIWLc49cvKyjdGS4YzhgjjFs2Vuc/08fnXU2gqUVIjIdmMbI9nUnrsAw/njdq1vN7es+FPzOQ/cxHuQcWjMw3hOQuXND6cMXTn4wC6SeRuRDQ+oNf/v0/ssRLK0SUYAsIzvIqaDJNAucTWt+1p24MuPsQ+dM+tq5uYCjqaNR6iGwMCewNyFYWiEiU2nuCKeCnA3kac1TnATxF36+dfrk+EWasIaPDl3t1AIFSEQm01wvp4K8j9b9mJNk+8HSo9NCS3P8lgLUChGZTnMDnAqymtb0dCfuBU6S1PlWYneTAtQOIjqC6+dUkItozTpOovWp2+Chh+Z+85LQdihA70VEJtFcmZNNCwgxcAGt+QEn35M0FwFXK0BjJaJ3QIOcbLK6xQUEB/yQNmrj0sPHFKDTj4gmILmE1jzdnbijnHxP8N6uGbrNuwkFSES6aC6kraSN08Ojp+hDeduAHS2s939YARoLEU1ACSeT3v98APgIrflvaiboMdwNClAzItIxYX4HpPh0Ao/Rmp90J65nIgbIGEgSx0DiFKCmRCTLKSe1teX1wExa8yDARAxQOfHkshErl8xbfvXMKWcpQCMRkTzNGcab4jMf2AicRWsc8AinUHfidgFv0sB7j/OeJcsWcfZ5Z7NwwdybFKBGIkJtSynPqaZ3PpuAM2nd/1QD0A8w0aagJPV0xCFdk7ro6+3HwXUK0HBExAAxzXnGi+KzFlgPTGV0/pGJ4UkaeCCKIqw1JElCmMlcfMuKRTMUoEYiErcwAXXRdlK7MfpHjN6e7sQ9xcTwDMMwxgAG7z1BGGKNuUYBaiQiIZChuYjxIL/HifkHJohqCPcAW6ljgDRNcc6BAWsNmWz2WgWokYjk3itArorxIDdyYr7JxPL08QFypM5hMTjnCTOZqxSgRiISAeHI8fEE1nraTcdv04ALGL3v1K7emUh+etxvgNKUtJJgjME7RxiFUz+1YtGlClA9EQkBwzC883jvWbhwbkK7yUWcmHuZeJ6ljrGGNHFUKhVsYPHeY8OAMIquVYDqiUjHSAEqO08+n2Xy5ElzaDc5i9H7XnfitjPBdCduJ7CNGgMkQLlcwRjDEA9RHF+tANUTEccIEmD27JlkO/Pn0G6ylNH7fSauZ+o34BxQLpWPBcg7R5iJ3n/z0vmzFaARiYgxMJg4ZnTmWLB4AUmSXnTzsgUR7STzGZ2/6U7cXiauZ2lQLJbw3gPgnCMIQzLZJlOQAiQildSTAZavXIY3gDGdYRReSTvJZFr3XDU+f0mDCb2IAJSKJdIkOTYFWWsJwnCtAjQsETFAyXkWLTyDKdOnUBgYJAhDokzmd2kn6aA1D1bjcxkTXDWQvwB2URMApVKJpJxgrT02BUVx5koFaFgiUkocU7IRcxfMo1gs1dZoU+Jc7rc/vmhOJyeLbAFurMbndAr/s9RYA6VSmVK5hA3eDVAYRdNvW7X0YgVIRAAMdVJg9hmzyOZzVMoVAFzqCOOMzXZ03EW7yEGG9wJwRzU851af/+X08hw11hoqDkqFEkEQMMR7bBBgw2AtvwZCRk1E4jiDDSz4ujx5R7Yz96fA39EOcg+QASYBe4BuYF01Ohs5fXXXb8KleAq1KbpelImuBu5TgESkhzqhgV3v7GXajKlk81mKhSLGGNIkJc5mp9+6aumnH9r61reRsb4z2QCs4VfLRuBo/YJFsVAE76mp3YoQffCW5YumPLxtx5Ff7yM4Ef0jPAC8QU0msBwZKPLqK69hvKcanWOrtN5DrjN/H8MTRTUBnqfG1gKUVBKMNfXr2Ka2VYkCJCL3UuOBfGjp6S+waePPsXhyHTm8hzRJiPO5ubeuWvJJhieynpoAKBaLlMtlrLXUYKwlCII1CpCI0J24/wB+TJ2O0LLvSB8vv7iZwb4Buro6yOWydORzTJ429R6aEwUIa6FcTigVy7VFhBo8URwrQMeIyFpgc2OEDvcPDkVo29Y32b+vh3d27ObtN7evuhBWcDyRF+ovJa34oSkIG1hqcKkjiMJFn1y2YCW/wgyjIaJPBETAY8BV1BgDlYqjAgSAYwhxaOc9n7g9HE/0d/QGsBygkDiWLZrLkjMXUxgsUIMNAvoPH/ncd7a8+YC24ESE7sRVgDXVEN0H3APgPYShJYRjywjGmIe7R46PyAZgef0mnHcOMIAHwABRHF8JPKAjuHoiCtEXgdXADxq/8199NgJ30owoQDW2dilpkiRYa6jB4wnC4PLqMVyoADUSUYQ2VJ/rgfOBe4F/B/6oGqf3V58CIxPZTE0AlEul6lPBWgv1X9oNwy4b2Et0BDcSEYVoE7CJ1olsadiEqz5lsrksx3hPEIWEUbQGeFYT0JiJiEjtq637AIwxJB5KxdqlpHW8hygb30AbKUAiIvJ6LUCkdV9HredcSpjJnF/9Sup8BahdRERkGzUeKJZK4D31vPOEYUgYZ25UgNpFRETebLwTLk3TxikIjCHOxp9QgNpFRER+QU1QC1C5VLsTro5LUzJxfOknlsybpQC1g4iIvE1NYKFYrAxFKAhD6nnnCDIRmTZNQQqQiIhsBxyAtZYyMDAwSBhaGuEhzuXuUIDaQUREq9g9wG5qPNDf1493nkYuTYmy8fnVWxGWK0DtICIiO6mJgL6jfbXfAwXU835oG45MNv60AtQOIiKyi5rQGvqLFfp6+8hkIhp574jzudsVoLYSEdEEZKwhAY4cOoK1Fk89SNOUKJOZ/akVi69SgMZKRETeoU4EHOo5THGwQBgG1MODCSzZjtznFKCxEhGRndSJQkNvOalG6BDZOMZ7Tz2XOqI4vq56Nc90BWgsRERkN/UwAOzbu580SY+/nNQ5okyGKM58VgEaKxERTUCOOtnAcLBvkAP7DpDL5RqnILz35Do77lKAxkJERL8F2g/8kjrWGgB27txNWkkIo5B6aZIQZeMzqssIaxWgthER0Xsg7yEbWg4NFNm1fRf5XI5G1gwtI/yFAjQWIiKygWHEFt7asYdDBw6S78jjva+fgsjkcr9x87IFSxSgEyUiIn8N7KVBFFgc8OqWNygVCsciVH8zQjaf+7wCdKJERPQeaC9wAfB641FcLrQMlBM2vLyFUqFIZ1cnxhi897g0JduRv+Pji+ZkOU0ZTjkREbkktDHwfeAq6hhgIHHko5CVq5Yyc9YMqEUosJaNL226dd2h3u8qQGMlIqIQ3Q/8IXWMgWLFATB7xlSmV59cR44jPYfZvnPPUy96rlCA2kFERBG6GfhXIEuNAZz3FFOPAUIgBaLAvLU+9csUoHYREVGEFgPfAtbQwHvw3mOtAfhZd+IuVYDaTUREIfoM8CVgJsO7qxqgbyhA40FERBHqAu4GbgNm867vVeNzO6ep/wd7k7QQLn/BgwAAAABJRU5ErkJggg==",
            };
        }
    }
}
